package configuration

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strings"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "extensions"
)

//go:generate go run github.com/maxbrunsfeld/counterfeiter/v6 -generate

//counterfeiter:generate . Client
// Client is the client library to the extensions configuration service
type Client interface {
	GetChannelConfiguration(ctx context.Context, channelID string, includeCommon bool, extensionIDs []string, reqOpts *twitchclient.ReqOpts) (protocol.RecordMap, error)
	SetChannelConfiguration(ctx context.Context, channelID, extensionID string, developer, broadcaster *protocol.Record, reqOpts *twitchclient.ReqOpts) error
	DeleteChannelConfiguration(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) error

	GetConfiguration(ctx context.Context, addresses []protocol.Address, reqOpts *twitchclient.ReqOpts) (protocol.RecordMap, error)
	SetConfiguration(ctx context.Context, address *protocol.Address, record *protocol.Record, reqOpts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	http twitchclient.Client
}

// NewClient creates an http client to call the configuration endpoints
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{
		http: twitchClient,
	}, nil
}

func (cli *clientImpl) GetChannelConfiguration(ctx context.Context, channelID string, includeCommon bool, extensionIDs []string, reqOpts *twitchclient.ReqOpts) (protocol.RecordMap, error) {
	req, err := cli.http.NewRequest(http.MethodGet, fmt.Sprintf("/channel/%s?ext_ids=%s&common=%t", channelID, strings.Join(extensionIDs, ","), includeCommon), nil)
	if err != nil {
		return nil, err
	}
	out := make(protocol.RecordMap)
	err = cli.execute(ctx, req, "get_channel_configuration", &out, reqOpts)
	return out, err
}

func createGetConfigurationBody(addresses []protocol.Address) io.Reader {
	out := &bytes.Buffer{}
	_ = json.NewEncoder(out).Encode(addresses)
	return out
}

func (cli *clientImpl) GetConfiguration(ctx context.Context, addresses []protocol.Address, reqOpts *twitchclient.ReqOpts) (protocol.RecordMap, error) {
	req, err := cli.http.NewRequest(http.MethodPost, "/addresses", createGetConfigurationBody(addresses))
	if err != nil {
		return nil, err
	}
	out := make(protocol.RecordMap)
	err = cli.execute(ctx, req, "get_configuration", &out, reqOpts)
	return out, err
}

func createSetConfigurationBody(address *protocol.Address, record *protocol.Record) io.Reader {
	out := &bytes.Buffer{}
	_ = json.NewEncoder(out).Encode(&protocol.SetConfigurationInput{
		Address: *address,
		Record:  record,
	})
	return out
}

func (cli *clientImpl) SetConfiguration(ctx context.Context, address *protocol.Address, record *protocol.Record, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest(http.MethodPut, "/addresses", createSetConfigurationBody(address, record))
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "set_configuration", nil, reqOpts)
	return err
}

func createSetChannelConfigurationBody(channelID, extensionID string, developer *protocol.Record, broadcaster *protocol.Record) io.Reader {
	out := &bytes.Buffer{}
	_ = json.NewEncoder(out).Encode(protocol.SetChannelConfigurationInput{
		ChannelID:   channelID,
		ExtensionID: extensionID,
		Developer:   developer,
		Broadcaster: broadcaster,
	})
	return out
}

func (cli *clientImpl) SetChannelConfiguration(ctx context.Context, channelID, extensionID string, developer *protocol.Record, broadcaster *protocol.Record, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest(http.MethodPut, fmt.Sprintf("/channel/%s", channelID), createSetChannelConfigurationBody(channelID, extensionID, developer, broadcaster))
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "set_channel_configuration", nil, reqOpts)
	return err
}

func (cli *clientImpl) DeleteChannelConfiguration(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) error {
	req, err := cli.http.NewRequest(http.MethodDelete, fmt.Sprintf("/channel/%s", channelID), nil)
	if err != nil {
		return err
	}
	err = cli.execute(ctx, req, "delete_channel_configuration", nil, reqOpts)
	return err
}

func (cli *clientImpl) execute(ctx context.Context, req *http.Request, statName string, out interface{}, reqOpts *twitchclient.ReqOpts) error {
	opts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions." + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	defer close(req.Body)

	resp, err := cli.http.Do(ctx, req, opts)
	if err != nil {
		return err
	}

	defer close(resp.Body)

	if err, found := errors.ExtractFromHTTPResponse(resp); found {
		return err
	}

	if resp.StatusCode == http.StatusNoContent {
		out = nil
		return nil
	}

	if err = json.NewDecoder(resp.Body).Decode(out); err != nil {
		return fmt.Errorf("Unable to read response body: %s", err)
	}
	return nil
}

func close(c io.Closer) {
	if c != nil {
		_ = c.Close()
	}
}
