package configuration

import (
	"context"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/controller"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	configuration "code.justin.tv/extensions/fulton-configuration/server"
	"code.justin.tv/extensions/fulton-configuration/server/api"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	goji "goji.io"
)

const (
	testChannel   = "chID"
	testExtension = "extID"
)

func createMockService(t *testing.T) (model.Store, *auth.FakeHandler, *httptest.Server) {
	store := memory.New(uuid.NewSource())
	ch1 := model.NewChannel(model.DefaultEnvironment, testExtension, testChannel)
	dev := "developer"
	ch1.Developer = protocol.NewRecord("1", &dev)
	require.NoError(t, store.SaveChannel(ch1))
	co1 := model.NewCommon(model.DefaultEnvironment, testExtension)
	gol := "global"
	co1.Global = protocol.NewRecord("", &gol)
	require.NoError(t, store.SaveCommon(co1))

	router := goji.NewMux()
	// TODO : use an auth handler that reads permissions per call
	authHandler := &auth.FakeHandler{Credentials: auth.NoPermissions()}
	router.Use(auth.StoreHandler(authHandler))
	router.Use(data.StoreManager(controller.New(store, event.NewCoordinator(false))))
	a := &api.API{}
	configuration.AppendFunctions(router, a)
	return store, authHandler, httptest.NewServer(router)
}

func TestClient_GetChannelConfiguration(t *testing.T) {
	_, _, server := createMockService(t)

	t.Run("should handle empty array", func(t *testing.T) {
		cli, err := NewClient(twitchclient.ClientConf{Host: server.URL})
		require.NoError(t, err)
		values, err := cli.GetChannelConfiguration(context.Background(), testChannel, true, []string{}, nil)
		assert.Empty(t, values)
		assert.NoError(t, err)
	})

	t.Run("should handle unset values", func(t *testing.T) {
		cli, err := NewClient(twitchclient.ClientConf{Host: server.URL})
		require.NoError(t, err)
		values, err := cli.GetChannelConfiguration(context.Background(), testChannel, true, []string{"not_used"}, nil)
		assert.Empty(t, values)
		assert.NoError(t, err)
	})

	t.Run("should return matches (ignoring common when asked)", func(t *testing.T) {
		cli, err := NewClient(twitchclient.ClientConf{Host: server.URL})
		require.NoError(t, err)

		values, err := cli.GetChannelConfiguration(context.Background(), testChannel, false, []string{testExtension}, nil)
		assert.Len(t, values, 1)
		assert.NoError(t, err)
	})

	t.Run("should handle common matches (when enabled)", func(t *testing.T) {
		cli, err := NewClient(twitchclient.ClientConf{Host: server.URL})
		require.NoError(t, err)

		values, err := cli.GetChannelConfiguration(context.Background(), testChannel, true, []string{testExtension}, nil)
		assert.Len(t, values, 2)
		assert.NoError(t, err)
	})
}

func TestClient_SetChannelConfiguration(t *testing.T) {

}

func TestClient_DeleteChannelConfiguration(t *testing.T) {

}

func TestClient_GetConfiguration(t *testing.T) {

}

func TestClient_SetConfiguration(t *testing.T) {

}
