package main

import (
	"fmt"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"log"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/gds/gds/golibs/sandstorm"
	"code.justin.tv/gds/gds/golibs/uuid"

	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/c7s"
	"code.justin.tv/extensions/fulton-configuration/data/controller"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/cached"
	"code.justin.tv/extensions/fulton-configuration/data/model/dynamo"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/listeners"
	configuration "code.justin.tv/extensions/fulton-configuration/server"
	"code.justin.tv/extensions/fulton-configuration/server/api"
	"golang.a2z.com/FultonGoLangBootstrap/bootstrap"
	"golang.a2z.com/FultonGoLangBootstrap/fultonecs"
	simplebootstrap "golang.a2z.com/FultonGoLangSimpleBootstrap/bootstrap"

	barbradyTwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
)

const (
	defaultHystrixTimeout = 3000
)

func createEventCoordinator(cp client.ConfigProvider, cfg c7s.Config) *event.Coordinator {
	coord := event.NewCoordinator(true)
	arn := cfg.EventsSnsArn
	if arn != "" {
		err := coord.Register(listeners.NewEventBusListener(sns.New(cp), cfg.AppEnv, arn))
		if err != nil {
			log.Printf("Failed to register Event Coordinator")
		}
	}

	return coord
}

func createMetrics(cfg c7s.Config) statsd.Statter {
	switch cfg.StatsMethod {
	case "noop":
		statter := &statsd.Client{}
		return statter
	case "statsd":
		prefix := fmt.Sprintf("%s.%s", cfg.ComponentName, cfg.AppEnv)
		statter, err := statsd.NewClientWithConfig(&statsd.ClientConfig{
			Address:       cfg.StatsdHost,
			Prefix:        prefix,
			ResInterval:   0,
			UseBuffered:   false,
			FlushInterval: time.Second,
			FlushBytes:    0,
		})

		if err != nil {
			panic(fmt.Errorf("Error creating statsd client: %v", err))
		}
		return statter
	default:
		panic(fmt.Errorf("Unknown stat method: %v", cfg.StatsMethod))
	}
}

func createDataCache(cfg c7s.Config, store model.Store) model.Store {
	duration, err := time.ParseDuration(cfg.StoreCacheDuration)
	if err != nil {
		panic(fmt.Errorf("Illegal Cache Duration specified: %v", err))
	}
	// TODO : support dynamic cache duration tuning -- this requires updating golibs
	return cached.New(store, duration)
}

func createDataStore(cp client.ConfigProvider, sessionCredentials *credentials.Credentials, cfg c7s.Config) model.Store {
	switch cfg.StoreMethod {
	case "memory":
		return memory.New(uuid.NewSource())
	case "dynamo":
		aggr, err := time.ParseDuration(cfg.DynamoAggregationTime)
		if err != nil {
			panic(fmt.Errorf("Illegal DynamoAggregationTime specified: %v", err))
		}
		return dynamo.New(uuid.NewSource(), dynamodb.New(cp, aws.NewConfig().WithCredentials(sessionCredentials)), nil, cfg.DynamoPrefix, aggr)
	default:
		panic(fmt.Errorf("Unknown storage method: %v", cfg.StoreMethod))
	}
}

func createAuthHandler(cfg c7s.Config, statter statsd.Statter) auth.Handler {
	switch cfg.AuthMethod {
	case "fake":
		// TODO : revisit ability to do this outside of dev environments.
		log.Printf("Installing a fake auth handler; should allow everything")
		return &auth.FakeHandler{Credentials: auth.AllPermissions()}
	case "cartman":
		log.Printf("Installing JWT/Cartman authentication")
		decoder, err := sandstorm.NewDecoder(
			cfg.SandstormRegion,
			cfg.SandstormRoleArn,
			cfg.SandstormTable,
			cfg.SandstormKeyId,
			cfg.CartmanSecret,
			auth.CartmanAudience,
			auth.CartmanIssuer)
		if err != nil {
			panic(fmt.Errorf("Unable to initialize Sandstorm: %v", err))
		}

		barbradyHostName := cfg.BarbradyHost
		barbradyClient := createBarbradyClient(barbradyHostName)
		barbradySamplingPercent := cfg.BarbradySamplingRate
		return auth.NewJWTHandler(decoder, barbradyClient, barbradySamplingPercent, statter)
	default:
		panic(fmt.Errorf("Unknown auth method: %v", cfg.AuthMethod))
	}
}

func createBarbradyClient(barbradyHostName string) barbradyTwirp.TwitchExtensionsBarbrady {
	return barbradyTwirp.NewTwitchExtensionsBarbradyProtobufClient(
		barbradyHostName,
		&http.Client{},
	)
}

func main() {
	// Spin up a Fulton bootstrap for ECS (via FultonGoLangBootstrap)
	bootstrapper := &fultonecs.ECSBootstrapper{
		ConfigSetLoader:          bootstrap.DefaultConfigSetLoader,
		DisableLoggingMiddleware: true,
		MetricsPlatform:          simplebootstrap.CloudWatch,
		ServiceName:              "TwitchExtensionsConfiguration",
	}

	bs, err := bootstrapper.Bootstrap()
	if err != nil {
		panic(fmt.Sprintf("Error bootstrapping fulton: %s", err))
	}

	// Load C7 server-side config
	config := c7s.Config{}
	err = bs.C7.FillWithNamespace("TwitchExtensionsConfiguration", &config)
	if err != nil {
		panic(fmt.Sprintf("Error loading config: %s", err))
	}

	// Create AWS session
	awsConfig := aws.NewConfig().WithRegion("us-west-2").WithCredentialsChainVerboseErrors(true)
	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		panic(err)
	}

	// Create cross account AWS session for Dynamo migration
	dynamoCrossAccountRole := stscreds.NewCredentials(awsSession, config.CrossAccountRoleArn, func(arp *stscreds.AssumeRoleProvider) {
		arp.RoleSessionName = "fulton-dynamo-access-configuration"
		arp.Duration = 60 * time.Minute
		arp.ExpiryWindow = 30 * time.Second
	})

	statter := createMetrics(config)
	defer statter.Close()

	handler := createAuthHandler(config, statter)
	// Currently uses cross-account session
	store := createDataStore(awsSession, dynamoCrossAccountRole, config)
	store = createDataCache(config, store)
	coord := createEventCoordinator(awsSession, config)
	mgr := controller.New(store, coord)
	a := api.NewAPI(bs.SampleReporter)

	router := configuration.BuildServer(defaultHystrixTimeout, handler, mgr, a)
	server := &http.Server{
		Addr:              config.Port,
		Handler:           router,
		ReadTimeout:       0,
		ReadHeaderTimeout: 0,
		WriteTimeout:      0,
		IdleTimeout:       0,
	}

	log.Println("Listening on " + config.Port + "...")
	err = server.ListenAndServe()
	if err != nil {
		log.Fatal(err)
	}
}
