package access

import (
	"sync"

	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/extensions/fulton-configuration/protocol/messages"
)

type channelForWrite struct {
	value   *model.Channel
	prev    *model.Channel
	store   model.Store
	pubLock sync.Mutex
}

func NewChannelForWrite(value *model.Channel, store model.Store) (RecordForWrite, error) {
	m, err := value.GetMemento()
	if err != nil {
		return nil, err
	}
	copy := &model.Channel{}
	if err = copy.ApplyMemento(m); err != nil {
		return nil, err
	}
	return &channelForWrite{value: copy, prev: value, store: store}, nil
}

func (c *channelForWrite) ChannelID() string { return c.value.ChannelID }

func (c *channelForWrite) Get(s protocol.Segment) (*protocol.Record, error) {
	if s.ChannelID() != c.value.ChannelID {
		return nil, protocol.ErrNonAtomicWrite
	}
	switch s.Type() {
	case protocol.BroadcasterType:
		return c.value.Broadcaster, nil
	case protocol.DeveloperType:
		return c.value.Developer, nil
	}
	return nil, protocol.ErrUnimplemented
}

func (c *channelForWrite) Set(s protocol.Segment, r *protocol.Record) error {
	if s.ChannelID() != c.value.ChannelID {
		return protocol.ErrNonAtomicWrite
	}
	switch s.Type() {
	case protocol.BroadcasterType:
		c.value.Broadcaster = r
		return nil
	case protocol.DeveloperType:
		c.value.Developer = r
		return nil
	}
	return protocol.ErrUnimplemented
}

func (c *channelForWrite) Save() error {
	// no changes detected
	if c.prev.MsgSeq == c.value.MsgSeq {
		return nil
	}
	return c.store.SaveChannel(c.value)
}

func (c *channelForWrite) IsUnpublished() bool {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	return c.isUnpublished()
}

func (c *channelForWrite) UnhandledTopics() []event.Topic {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() {
		return []event.Topic{messages.ConfigTopic}
	}
	return []event.Topic{}
}

func (c *channelForWrite) UnhandledMessages(topic event.Topic) []event.Message {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() && topic == messages.ConfigTopic {
		return c.value.Messages
	}
	return []event.Message{}
}

func (c *channelForWrite) MarkHandled(topic event.Topic) error {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() && topic == messages.ConfigTopic {
		if err := c.store.MarkChannelPublished(c.value); err != nil {
			return err
		}
		c.value.UnpublishedTime = nil
		c.value.Messages = []event.Message{}
	}
	return nil
}

func (c *channelForWrite) isUnpublished() bool {
	return len(c.value.Messages) > 0
}

func (c *channelForWrite) UpdateMessages(creds auth.Credentials) error {
	if !protocol.RecordEqual(c.value.Broadcaster, c.prev.Broadcaster) {
		s, err := protocol.Broadcaster(c.value.ChannelID)
		if err != nil {
			return err
		}
		msg := c.buildMessage(creds, s, c.value.Broadcaster)
		c.value.Messages = append(c.value.Messages, msg)
	}
	if !protocol.RecordEqual(c.value.Developer, c.prev.Developer) {
		s, err := protocol.Developer(c.value.ChannelID)
		if err != nil {
			return err
		}
		msg := c.buildMessage(creds, s, c.value.Developer)
		c.value.Messages = append(c.value.Messages, msg)
	}
	return nil
}

func (c *channelForWrite) buildMessage(creds auth.Credentials, s protocol.Segment, r *protocol.Record) event.Message {
	c.value.MsgSeq++
	msg := messages.NewConfigMessage(c.value.MsgSeq, messages.OnSet)
	if creds.UserID() != nil {
		msg.UserID = *creds.UserID()
	}
	msg.Record = r
	msg.Address = s.Address(c.value.ExtensionID)
	return msg
}
