package access

import (
	"sync"

	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/extensions/fulton-configuration/protocol/messages"
)

type commonForWrite struct {
	value   *model.Common
	prev    *model.Common
	store   model.Store
	pubLock sync.Mutex
}

func NewCommonForWrite(value *model.Common, store model.Store) (RecordForWrite, error) {
	m, err := value.GetMemento()
	if err != nil {
		return nil, err
	}
	copy := &model.Common{}
	if err = copy.ApplyMemento(m); err != nil {
		return nil, err
	}
	return &commonForWrite{value: copy, prev: value, store: store}, nil
}

func (*commonForWrite) ChannelID() string { return "" }

func (c *commonForWrite) Get(s protocol.Segment) (*protocol.Record, error) {
	if s.ChannelID() != "" {
		return nil, protocol.ErrNonAtomicWrite
	}
	switch s.Type() {
	case protocol.GlobalType:
		return c.value.Global, nil
	}
	return nil, protocol.ErrUnimplemented
}

func (c *commonForWrite) Set(s protocol.Segment, r *protocol.Record) error {
	if s.ChannelID() != "" {
		return protocol.ErrNonAtomicWrite
	}
	switch s.Type() {
	case protocol.GlobalType:
		c.value.Global = r
		return nil
	}
	return protocol.ErrUnimplemented
}

func (c *commonForWrite) Save() error {
	// no changes detected
	if c.prev.MsgSeq == c.value.MsgSeq {
		return nil
	}
	return c.store.SaveCommon(c.value)
}

func (c *commonForWrite) IsUnpublished() bool {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	return c.isUnpublished()
}

func (c *commonForWrite) UnhandledTopics() []event.Topic {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() {
		return []event.Topic{messages.ConfigTopic}
	}
	return []event.Topic{}
}

func (c *commonForWrite) UnhandledMessages(topic event.Topic) []event.Message {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() && topic == messages.ConfigTopic {
		return c.value.Messages
	}
	return []event.Message{}
}

func (c *commonForWrite) MarkHandled(topic event.Topic) error {
	defer c.pubLock.Unlock()
	c.pubLock.Lock()
	if c.isUnpublished() && topic == messages.ConfigTopic {
		if err := c.store.MarkCommonPublished(c.value); err != nil {
			return err
		}
		c.value.UnpublishedTime = nil
		c.value.Messages = []event.Message{}
	}
	return nil
}

func (c *commonForWrite) isUnpublished() bool {
	return len(c.value.Messages) > 0
}

func (c *commonForWrite) UpdateMessages(creds auth.Credentials) error {
	if !protocol.RecordEqual(c.value.Global, c.prev.Global) {
		msg := c.buildMessage(creds, protocol.Global(), c.value.Global)
		c.value.Messages = append(c.value.Messages, msg)
	}
	return nil
}

func (c *commonForWrite) buildMessage(creds auth.Credentials, s protocol.Segment, r *protocol.Record) event.Message {
	c.value.MsgSeq++
	msg := messages.NewConfigMessage(c.value.MsgSeq, messages.OnSet)
	if creds.UserID() != nil {
		msg.UserID = *creds.UserID()
	}
	msg.Record = r
	msg.Address = s.Address(c.value.ExtensionID)
	return msg
}
