package controller

import (
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/controller/access"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"log"
)

// type segmentMap = map[protocol.Segment]*protocol.Record

type extConfig struct {
	env      string
	extID    string
	mgr      *manager
	forWrite access.RecordForWrite
	err      error
}

var _ data.ExtensionConfiguration = (*extConfig)(nil)

func newExtConfig(mgr *manager, env, extID string) (data.ExtensionConfiguration, error) {
	if extID == "" {
		return nil, protocol.ErrMissingExtensionID
	}
	return &extConfig{
		env:   env,
		extID: extID,
		mgr:   mgr,
	}, nil
}

func (e *extConfig) Get(segments ...protocol.Segment) (protocol.RecordMap, error) {
	out, err := e.mgr.Get(protocol.AddressesForSegments(segments, e.extID)...)
	if err == nil {
		err = e.override(segments, out)
	}
	return out, err
}

func (e *extConfig) Set(s protocol.Segment, r *protocol.Record) data.ExtensionConfiguration {
	if e.err != nil {
		return e
	}

	if !auth.Load(e.mgr.ctx).CanEditConfig(e.extID, s) {
		e.err = protocol.ErrUnauthorized
		return e
	}

	length := r.Length()
	if length > protocol.MaxSegmentLength {
		e.err = protocol.ErrIllegalSegmentLength(s.Address(e.extID), length)
		return e
	}

	if e.forWrite == nil {
		e.forWrite, e.err = e.createForWrite(s, r)
		if e.err != nil {
			return e
		}
	}

	e.err = e.forWrite.Set(s, r)
	return e
}

func (e *extConfig) Save() error {
	if e.err != nil {
		return e.err
	}
	if e.forWrite == nil {
		return nil
	}
	e.err = e.forWrite.UpdateMessages(auth.Load(e.mgr.ctx))
	if e.err == nil {
		e.err = e.forWrite.Save()
	}
	if e.err == nil {
		err := e.mgr.coord.OnUpdate(e.forWrite)
		if err != nil {
			log.Println("extension_configuration Save OnUpdate failed with: " + err.Error())
		}
	}
	return e.err
}

func (e *extConfig) createForWrite(s protocol.Segment, r *protocol.Record) (access.RecordForWrite, error) {
	if s.ChannelID() == "" {
		c, err := e.mgr.store.LoadCommon(e.env, e.extID)
		if err != nil {
			return nil, err
		}
		if c == nil {
			c = model.NewCommon(e.env, e.extID)
		}
		return access.NewCommonForWrite(c, e.mgr.store)
	}
	c, err := e.mgr.store.LoadChannel(e.env, e.extID, s.ChannelID())
	if err != nil {
		return nil, err
	}
	if c == nil {
		c = model.NewChannel(e.env, e.extID, s.ChannelID())
	}
	return access.NewChannelForWrite(c, e.mgr.store)
}

func (e *extConfig) override(segments []protocol.Segment, out protocol.RecordMap) error {
	if e.forWrite != nil {
		for _, s := range segments {
			if s.ChannelID() == e.forWrite.ChannelID() {
				r, err := e.forWrite.Get(s)
				if err != nil {
					return err
				}
				a := s.Address(e.extID)
				if r == nil {
					delete(out, a)
				} else {
					out[a] = r
				}
			}
		}
	}
	return nil
}
