package controller

import (
	"context"

	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

type manager struct {
	store model.Store
	coord *event.Coordinator
	ctx   context.Context
}

func New(store model.Store, coord *event.Coordinator) data.ConfigurationManager {
	return &manager{store, coord, context.Background()}
}

func (m *manager) WithContext(ctx context.Context) data.ConfigurationManager {
	return &manager{m.store, m.coord, ctx}
}

func (m *manager) Get(addresses ...protocol.Address) (protocol.RecordMap, error) {
	promises := []model.RecordPromise{}
	for _, addr := range addresses {
		if addr.ChannelID == "" {
			promises = append(promises, m.store.AsyncLoadCommon(model.DefaultEnvironment, addr.ExtensionID).ForSegment(addr.SegmentType))
		} else {
			promises = append(promises, m.store.AsyncLoadChannel(model.DefaultEnvironment, addr.ExtensionID, addr.ChannelID).ForSegment(addr.SegmentType))
		}
	}
	out := make(protocol.RecordMap)
	for _, p := range promises {
		addr, rec, err := p.Record()
		if err != nil {
			return nil, err
		}
		if addr != nil && rec != nil {
			out[*addr] = rec
		}
	}
	return out, nil
}

func (m *manager) GetByChannel(channelID string, includeCommon bool, extensionIDs ...string) (protocol.RecordMap, error) {
	segments, err := protocol.SegmentsForChannel(channelID, includeCommon)
	if err != nil {
		return nil, err
	}
	addr := []protocol.Address{}
	for _, extID := range extensionIDs {
		addr = append(addr, protocol.AddressesForSegments(segments, extID)...)
	}
	return m.Get(addr...)
}

func (m *manager) ForExtension(extensionID string) (data.ExtensionConfiguration, error) {
	if extensionID == "" {
		return nil, protocol.ErrMissingExtensionID
	}
	return newExtConfig(m, model.DefaultEnvironment, extensionID)
}

func (m *manager) DeleteByChannel(channelID string) error {
	if channelID == "" {
		return protocol.ErrMissingChannelID
	}
	if !auth.Load(m.ctx).CanDeleteConfig(channelID) {
		return protocol.ErrUnauthorized
	}
	return m.store.DeleteChannel(channelID)
}
