package controller

import (
	"testing"

	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/bad"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

func TestManager_Get(t *testing.T) {
	segs, err := protocol.SegmentsForChannel("chID", true)
	require.NoError(t, err)

	t.Run("should request nothing when asked for nothing", func(t *testing.T) {
		mgr := New(bad.New(), event.NewCoordinator(false))
		out, err := mgr.Get()
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should return nothing when empty", func(t *testing.T) {
		mgr := New(memory.New(uuid.NewSource()), event.NewCoordinator(false))
		out, err := mgr.Get(protocol.AddressesForSegments(segs, "extID")...)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should return partial list when found", func(t *testing.T) {
		store := memory.New(uuid.NewSource())

		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		co := model.NewCommon(model.DefaultEnvironment, "extID")
		co.Global = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveCommon(co))

		mgr := New(store, event.NewCoordinator(false))

		addr := protocol.AddressesForSegments(segs, "extID")
		out, err := mgr.Get(addr...)
		assert.Len(t, out, 2) // finds broadcaster and global
		assert.NoError(t, err)
	})

	t.Run("should forward errors", func(t *testing.T) {
		mgr := New(bad.New(), event.NewCoordinator(false))

		addr := protocol.AddressesForSegments(segs, "extID")
		out, err := mgr.Get(addr...)
		assert.Empty(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})
}

func TestManager_GetByChannel(t *testing.T) {
	t.Run("should request nothing when asked for nothing", func(t *testing.T) {
		mgr := New(bad.New(), event.NewCoordinator(false))
		out, err := mgr.GetByChannel("chID", true)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should return nothing when empty", func(t *testing.T) {
		mgr := New(memory.New(uuid.NewSource()), event.NewCoordinator(false))
		out, err := mgr.GetByChannel("chID", true, "extID")
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should return partial list when found", func(t *testing.T) {
		store := memory.New(uuid.NewSource())

		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		co := model.NewCommon(model.DefaultEnvironment, "extID2")
		co.Global = protocol.NewRecord("3", nil)
		require.NoError(t, store.SaveCommon(co))

		mgr := New(store, event.NewCoordinator(false))
		out, err := mgr.GetByChannel("chID", true, "extID", "extID2")
		assert.Len(t, out, 2) // finds broadcaster for extID and global for extID2
		assert.NoError(t, err)
	})

	t.Run("should forward errors", func(t *testing.T) {
		mgr := New(bad.New(), event.NewCoordinator(false))
		out, err := mgr.GetByChannel("chID", true, "extID")
		assert.Empty(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})

	t.Run("should error when asked for an empty channel id", func(t *testing.T) {
		mgr := New(memory.New(uuid.NewSource()), event.NewCoordinator(false))
		out, err := mgr.GetByChannel("", true)
		assert.Empty(t, out)
		assert.Equal(t, protocol.ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
	})
}

func TestManager_ForExtension(t *testing.T) {
	mgr := New(bad.New(), event.NewCoordinator(false))
	out, err := mgr.ForExtension("extID")
	require.NoError(t, err) // don't access store in this call
	cast, ok := out.(*extConfig)
	require.True(t, ok)
	assert.Equal(t, "extID", cast.extID)
	assert.Equal(t, mgr, cast.mgr)
}
