package model

import (
	"strings"
	"time"

	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

// Channel represents the configuration for an extension and channel pair
// qualified by an environment
type Channel struct {
	Environment string
	ExtensionID string
	ChannelID   string
	Broadcaster *protocol.Record
	Developer   *protocol.Record
	// begin consistency enforcement
	ConcurrencyUUID string
	MsgSeq          int // used when creating messages
	Messages        []event.Message
	UnpublishedTime *time.Time
}

var _ lazy.MigratableWithMemento = (*Channel)(nil)

func (*Channel) FormatVersion() int  { return 1 }
func (*Channel) StoreAsVersion() int { return 1 }
func (*Channel) Migrations() (lazy.MigrationMap, lazy.MigrationMap) {
	return lazy.MigrationMap{}, lazy.MigrationMap{}
}

func NewChannel(env, extID, chID string) *Channel {
	return &Channel{
		Environment: env,
		ExtensionID: extID,
		ChannelID:   chID,
		Messages:    []event.Message{},
	}
}

type channelMemento struct {
	ChannelID   string
	EnvAndExtID string
	Broadcaster *recordForStorage `dynamodbav:",omitempty"`
	Developer   *recordForStorage `dynamodbav:",omitempty"`
	// begin consistency enforcement
	ConcurrencyUUID string
	MsgSeq          int        // used when creating messages
	Messages        []string   `dynamodbav:",omitempty"`
	UnpublishedTime *time.Time `dynamodbav:",omitempty"`
}

func (c *Channel) GetMemento() (interface{}, error) {
	msgs, err := event.Serialize(c.Messages...)
	if err != nil {
		return nil, err
	}
	return &channelMemento{
		c.ChannelID,
		c.Environment + ":" + c.ExtensionID,
		toStorage(c.Broadcaster),
		toStorage(c.Developer),
		c.ConcurrencyUUID,
		c.MsgSeq,
		msgs,
		c.UnpublishedTime,
	}, nil
}

func (c *Channel) ApplyMemento(in interface{}) error {
	proxy, ok := in.(*channelMemento)
	if !ok {
		return lazy.ErrMigrationCorrupt
	}
	segments := strings.Split(proxy.EnvAndExtID, ":")
	if len(segments) == 2 {
		c.Environment = segments[0]
		c.ExtensionID = segments[1]
	}
	c.ChannelID = proxy.ChannelID
	c.Broadcaster = fromStorage(proxy.Broadcaster)
	c.Developer = fromStorage(proxy.Developer)
	c.ConcurrencyUUID = proxy.ConcurrencyUUID
	var err error
	if c.Messages, err = event.Deserialize(messageFactory, proxy.Messages...); err != nil {
		return err
	}
	if c.Messages == nil {
		c.Messages = []event.Message{}
	}
	c.MsgSeq = proxy.MsgSeq
	c.UnpublishedTime = proxy.UnpublishedTime
	return nil
}
