package model

import (
	"time"

	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

// Common represents the configuration for an extension across all channels
// qualified by an environment
type Common struct {
	Environment string
	ExtensionID string
	Global      *protocol.Record
	// begin consistency enforcement
	ConcurrencyUUID string
	MsgSeq          int // used when creating messages
	Messages        []event.Message
	UnpublishedTime *time.Time
}

var _ lazy.MigratableWithMemento = (*Common)(nil)

func (*Common) FormatVersion() int  { return 1 }
func (*Common) StoreAsVersion() int { return 1 }
func (*Common) Migrations() (lazy.MigrationMap, lazy.MigrationMap) {
	return lazy.MigrationMap{}, lazy.MigrationMap{}
}

func NewCommon(env, ext string) *Common {
	return &Common{
		Environment: env,
		ExtensionID: ext,
		Messages:    []event.Message{},
	}
}

type commonMemento struct {
	Environment string
	ExtensionID string
	Global      *recordForStorage `dynamodbav:",omitempty"`
	// begin consistency enforcement
	ConcurrencyUUID string
	MsgSeq          int        // used when creating messages
	Messages        []string   `dynamodbav:",omitempty"`
	UnpublishedTime *time.Time `dynamodbav:",omitempty"`
}

func (c *Common) GetMemento() (interface{}, error) {
	msgs, err := event.Serialize(c.Messages...)
	if err != nil {
		return nil, err
	}
	return &commonMemento{
		c.Environment,
		c.ExtensionID,
		toStorage(c.Global),
		c.ConcurrencyUUID,
		c.MsgSeq,
		msgs,
		c.UnpublishedTime,
	}, nil
}

func (c *Common) ApplyMemento(in interface{}) error {
	proxy, ok := in.(*commonMemento)
	if !ok {
		return lazy.ErrMigrationCorrupt
	}
	c.Environment = proxy.Environment
	c.ExtensionID = proxy.ExtensionID
	c.Global = fromStorage(proxy.Global)
	c.ConcurrencyUUID = proxy.ConcurrencyUUID
	var err error
	if c.Messages, err = event.Deserialize(messageFactory, proxy.Messages...); err != nil {
		return err
	}
	if c.Messages == nil {
		c.Messages = []event.Message{}
	}
	c.MsgSeq = proxy.MsgSeq
	c.UnpublishedTime = proxy.UnpublishedTime
	return nil
}
