package model

import (
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
	"code.justin.tv/gds/gds/golibs/event"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/extensions/fulton-configuration/protocol/messages"
)

func TestCommon_Memento(t *testing.T) {
	start, err := time.Parse(time.RFC3339, "1999-12-31T23:59:59Z")
	require.NoError(t, err)

	message := messages.NewConfigMessage(1, messages.OnSet)
	message.Time = start
	content := "values"

	t.Run("should copy new values", func(t *testing.T) {
		ch := NewCommon("", "")
		m, err := ch.GetMemento()
		require.NoError(t, err)

		var ch2 Common
		assert.NoError(t, ch2.ApplyMemento(m))
		assert.Equal(t, *ch, ch2)
	})

	t.Run("should preserve values", func(t *testing.T) {
		ch := NewCommon("env", "extID")
		ch.Global = protocol.NewRecord("1", &content)
		ch.ConcurrencyUUID = "1234"
		ch.MsgSeq = -42
		ch.Messages = []event.Message{message}
		ch.UnpublishedTime = &start
		m, err := ch.GetMemento()
		require.NoError(t, err)

		var ch2 Common
		assert.NoError(t, ch2.ApplyMemento(m))
		assert.Equal(t, *ch, ch2)
	})

	t.Run("should error when given the wrong type", func(t *testing.T) {
		var ch Common
		assert.Equal(t, lazy.ErrMigrationCorrupt, ch.ApplyMemento("broken"))
	})

	t.Run("should forward bad message errors on get", func(t *testing.T) {
		ch := NewCommon("env", "extID")
		ch.Messages = []event.Message{&badMessage{}}
		m, err := ch.GetMemento()
		assert.Nil(t, m)
		assert.IsType(t, &json.MarshalerError{}, err)
	})

	t.Run("should forward bad message errors on apply", func(t *testing.T) {
		ch := NewCommon("env", "extID")
		m, err := ch.GetMemento()
		require.NoError(t, err)
		m.(*commonMemento).Messages = []string{"{"}
		var ch2 Common
		assert.Equal(t, event.ErrMalformedMessage, ch2.ApplyMemento(m))
	})
}

func TestCommon_Migration(t *testing.T) {
	start, err := time.Parse(time.RFC3339, "1999-12-31T23:59:59Z")
	require.NoError(t, err)

	message := messages.NewConfigMessage(1, messages.OnSet)
	message.Time = start
	content := "blah"
	src := &Common{
		Environment:     "env",
		ExtensionID:     "extID",
		Global:          protocol.NewRecord("1", &content),
		Messages:        []event.Message{message},
		MsgSeq:          1,
		UnpublishedTime: &start,
	}

	av, err := lazy.Marshal(src)
	assert.NoError(t, err)
	require.NotNil(t, av)

	dest := &Common{}
	assert.NoError(t, lazy.Unmarshal(av, dest))
	assert.EqualValues(t, src, dest)
}
