package harness

import (
	"testing"

	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"code.justin.tv/extensions/fulton-configuration/data/model"
)

// BlockTrackerGenerator is an input for the test harness. Each call should generate
// a tracker that doesn't collide with the others. It is expected that each call
// to this function returns a clean slate; if RunTest() allowParallel=false
// this function can clean up and return the same instance each time.
type BlockTrackerGenerator func(uuid uuid.Source) model.BlockTracker

type BlockTracker struct{}

// RunTests checks a store implementation for compliance with the reference
// implementation
func (b BlockTracker) RunTests(gen BlockTrackerGenerator, allowParallel bool, t *testing.T) {
	src := new(fakeUUID)
	b.checkInitialState(gen(src))(t)
	b.checkMark(gen(src))(t)
	b.checkUnmark(gen(src))(t)
	b.checkOnFinished(gen(src))(t)
}

func (b *BlockTracker) checkInitialState(tracker model.BlockTracker) func(t *testing.T) {
	return func(t *testing.T) {
		list, err := tracker.DeletionInProgress()
		assert.Empty(t, list)
		assert.NoError(t, err)

		marked, err := tracker.IsBlocked("chID").Active()
		assert.False(t, marked)
		assert.NoError(t, err)
	}
}

func (b *BlockTracker) checkMark(tracker model.BlockTracker) func(t *testing.T) {
	return func(t *testing.T) {
		marked, err := tracker.IsBlocked("chID").Active()
		assert.False(t, marked)
		assert.NoError(t, err)

		assert.NoError(t, tracker.Block("chID"))
		assert.NoError(t, tracker.Block("chID")) // idempotence

		marked, err = tracker.IsBlocked("chID").Active()
		assert.True(t, marked)
		assert.NoError(t, err)

		list, err := tracker.DeletionInProgress()
		assert.Equal(t, []string{"chID"}, list)
		assert.NoError(t, err)
	}
}

func (b *BlockTracker) checkUnmark(tracker model.BlockTracker) func(t *testing.T) {
	return func(t *testing.T) {
		require.NoError(t, tracker.Block("chID"))
		marked, err := tracker.IsBlocked("chID").Active()
		assert.True(t, marked)
		assert.NoError(t, err)

		assert.NoError(t, tracker.Unblock("chID"))
		assert.NoError(t, tracker.Unblock("chID")) // idempotence

		marked, err = tracker.IsBlocked("chID").Active()
		assert.False(t, marked)
		assert.NoError(t, err)

		list, err := tracker.DeletionInProgress()
		assert.Empty(t, list)
		assert.NoError(t, err)
	}
}

func (b *BlockTracker) checkOnFinished(tracker model.BlockTracker) func(t *testing.T) {
	return func(t *testing.T) {
		marked, err := tracker.IsBlocked("chID").Active()
		assert.False(t, marked)
		assert.NoError(t, err)

		// should not mark an item that was unmarked asynchronously
		assert.NoError(t, tracker.OnDeletionFinished("chID"))
		marked, err = tracker.IsBlocked("chID").Active()
		assert.False(t, marked)
		assert.NoError(t, err)

		require.NoError(t, tracker.Block("chID"))
		assert.NoError(t, tracker.OnDeletionFinished("chID"))

		marked, err = tracker.IsBlocked("chID").Active()
		assert.True(t, marked)
		assert.NoError(t, err)

		list, err := tracker.DeletionInProgress()
		assert.Empty(t, list)
		assert.NoError(t, err)
	}
}
