package shared

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

type mockEntry struct {
	addr *protocol.Address
	rec  *protocol.Record
	err  error
}
type mockMap map[protocol.SegmentType]mockEntry
type mockSrc struct {
	blocks bool
	items  mockMap
}

func (m *mockSrc) WouldBlock() bool { return m.blocks }

func (m *mockSrc) ForSegment(s protocol.SegmentType) model.RecordPromise {
	return &recordPromise{m, s}
}

func (m *mockSrc) RecordMap() (protocol.RecordMap, error) {
	out := make(protocol.RecordMap)
	for _, v := range m.items {
		out[*v.addr] = v.rec
	}
	return out, nil
}

func (m *mockSrc) Get() (interface{}, error) {
	return m.RecordMap()
}

func (m *mockSrc) find(s protocol.SegmentType) (*protocol.Address, *protocol.Record, error) {
	if v, ok := m.items[s]; ok {
		return v.addr, v.rec, v.err
	}
	return nil, nil, nil
}

func TestRecordPromise(t *testing.T) {
	t.Run("should handle blocking properly", func(t *testing.T) {
		p := &mockSrc{blocks: true}
		rp := p.ForSegment(protocol.BroadcasterType)
		assert.True(t, rp.WouldBlock())
	})

	t.Run("should handle missing entries", func(t *testing.T) {
		p := &mockSrc{items: mockMap{}}
		rp := p.ForSegment(protocol.BroadcasterType)
		assert.False(t, rp.WouldBlock())
		m, err := rp.Get()
		assert.Empty(t, m)
		assert.NoError(t, err)

		a, r, err := rp.Record()
		assert.Nil(t, a)
		assert.Nil(t, r)
		assert.NoError(t, err)
	})

	t.Run("should handle included entries", func(t *testing.T) {
		ba := &protocol.Address{
			ExtensionID: "extID",
			SegmentType: protocol.BroadcasterType,
			ChannelID:   "chID",
		}
		content := "record"
		br := protocol.NewRecord("1", &content)
		da := &protocol.Address{
			ExtensionID: "extID",
			SegmentType: protocol.DeveloperType,
			ChannelID:   "chID",
		}
		dr := protocol.NewRecord("1", &content)
		ge := protocol.ErrIllegalSegmentChannel(protocol.GlobalType, "chID")
		p := &mockSrc{items: mockMap{
			protocol.BroadcasterType: mockEntry{addr: ba, rec: br},
			protocol.DeveloperType:   mockEntry{addr: da, rec: dr},
			protocol.GlobalType:      mockEntry{err: ge},
		}}
		rp := p.ForSegment(protocol.BroadcasterType)
		assert.False(t, rp.WouldBlock())
		m, err := rp.Get()
		assert.Equal(t, protocol.RecordMap{*ba: br}, m)
		assert.NoError(t, err)

		a, r, err := rp.Record()
		assert.Equal(t, ba, a)
		assert.Equal(t, br, r)
		assert.NoError(t, err)

		rp = p.ForSegment(protocol.DeveloperType)
		assert.False(t, rp.WouldBlock())
		m, err = rp.Get()
		assert.Equal(t, protocol.RecordMap{*da: dr}, m)
		assert.NoError(t, err)

		a, r, err = rp.Record()
		assert.Equal(t, da, a)
		assert.Equal(t, dr, r)
		assert.NoError(t, err)

		rp = p.ForSegment(protocol.GlobalType)
		assert.False(t, rp.WouldBlock())
		m, err = rp.Get()
		assert.Empty(t, m)
		assert.Equal(t, ge, err)

		a, r, err = rp.Record()
		assert.Nil(t, a)
		assert.Nil(t, r)
		assert.Equal(t, ge, err)
	})
}
