package protocol

import (
	"testing"

	"code.justin.tv/gds/gds/golibs/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAddress(t *testing.T) {
	t.Run("should convert to appropriate segments when valid", func(t *testing.T) {
		seg, err := (&Address{SegmentType: GlobalType}).Segment()
		cmp := Global()
		assert.Equal(t, cmp, seg)
		assert.NoError(t, err)

		seg, err = (&Address{SegmentType: DeveloperType, ChannelID: "chID"}).Segment()
		cmp, _ = Developer("chID")
		assert.Equal(t, cmp, seg)
		assert.NoError(t, err)

		seg, err = (&Address{SegmentType: BroadcasterType, ChannelID: "chID"}).Segment()
		cmp, _ = Broadcaster("chID")
		assert.Equal(t, cmp, seg)
		assert.NoError(t, err)
	})

	t.Run("should error when given an invalid segment", func(t *testing.T) {
		seg, err := (&Address{SegmentType: SegmentType("???")}).Segment()
		assert.Nil(t, seg)
		assert.Equal(t, ErrUnknownSegmentTypeCode, errors.GetErrorCode(err))
	})

	t.Run("should error when given an invalid channel", func(t *testing.T) {
		seg, err := (&Address{SegmentType: GlobalType, ChannelID: "1"}).Segment()
		assert.Nil(t, seg)
		assert.Equal(t, ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))

		seg, err = (&Address{SegmentType: DeveloperType}).Segment()
		assert.Nil(t, seg)
		assert.Equal(t, ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))

		seg, err = (&Address{SegmentType: BroadcasterType}).Segment()
		assert.Nil(t, seg)
		assert.Equal(t, ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
	})
}

func TestParseAddress(t *testing.T) {
	t.Run("should support global by default", func(t *testing.T) {
		assert.Equal(t, ParseAddress("extID"), Address{
			ExtensionID: "extID",
			SegmentType: GlobalType,
		})
	})

	t.Run("should support common", func(t *testing.T) {
		assert.Equal(t, ParseAddress("extID:global"), Address{
			ExtensionID: "extID",
			SegmentType: GlobalType,
		})
	})

	t.Run("should support channel types", func(t *testing.T) {
		assert.Equal(t, ParseAddress("extID:broadcaster:chID"), Address{
			ExtensionID: "extID",
			SegmentType: BroadcasterType,
			ChannelID:   "chID",
		})

		assert.Equal(t, ParseAddress("extID:developer:chID"), Address{
			ExtensionID: "extID",
			SegmentType: DeveloperType,
			ChannelID:   "chID",
		})
	})
}

func TestAddressesForSegments(t *testing.T) {
	t.Run("should retain all information in a round trip", func(t *testing.T) {
		dev, _ := Developer("chID")
		bc, _ := Broadcaster("chID")
		segs := []Segment{Global(), dev, bc}
		addr := AddressesForSegments(segs, "extID", "extID2")
		assert.Len(t, addr, len(segs)*2)
		for i, a := range addr {
			out, err := a.Segment()
			require.NoError(t, err)
			assert.Equal(t, segs[i%len(segs)], out)
		}
	})
}
