package api

import (
	"bytes"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/controller"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/errors/errconv"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"context"
	"encoding/json"
	"io"
	"io/ioutil"
	"strings"
)

const (
	fakeUserId   = "FakeUserId"
	fakeJwt      = "FakeJWT"
	fakeClientId = "FakeClientId"
)

func createBody(ctx context.Context, input interface{}) context.Context {
	body := &bytes.Buffer{}
	_ = json.NewEncoder(body).Encode(input)

	return params.SetBody(ctx, ioutil.NopCloser(body))
}

func createManager(store model.Store) data.ConfigurationManager {
	if store == nil {
		store = memory.New(uuid.NewSource())
	}
	return controller.New(store, event.NewCoordinator(true))
}

func createEditConfigBody(bodyType string) (io.ReadCloser, error) {
	var data protocol.EditConfigurationV2Input
	switch bodyType {
	case "fullBody":
		data.ExtJwt = fakeJwt
		data.ClientId = fakeClientId
		data.UserID = fakeUserId
	case "jwtBody":
		data.ExtJwt = fakeJwt
		data.ClientId = fakeClientId
	case "userIdBody":
		data.UserID = fakeUserId
	case "missingClientId":
		data.ExtJwt = fakeJwt
	case "noBody":
	default:
		break
	}

	m, err := json.Marshal(data)
	if err != nil {
		return nil, err
	}
	reader := strings.NewReader(string(m))
	return ioutil.NopCloser(reader), nil
}

func convertAndReportResponse(ctx context.Context, route string, err error, reporter telemetry.SampleReporter) {
	if err != nil {
		convErr := errconv.LoadConverter(ctx).Convert(err)
		reporter.Report(route+" responses", float64(errors.GetHTTPStatusOrDefault(convErr, 500)), telemetry.UnitCount)
	} else {
		reporter.Report(route+" responses", float64(200), telemetry.UnitCount)
	}
}
