package api

import (
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"context"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/bad"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"testing"
)

func TestDeleteConfigurationV2(t *testing.T) {
	a := &API{}
	t.Run("should report unimplemented without a manager", func(t *testing.T) {
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx := params.SetBody(context.Background(), fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should report if channel is missing", func(t *testing.T) {
		ctx := data.Store(context.Background(), createManager(nil))
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingChannelID, err)
	})

	t.Run("should report unauthorized when applicable", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = data.Store(ctx, createManager(nil))
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnauthorized, err)
	})

	t.Run("should succeed if already empty", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(nil))
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not empty with fullBody", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Developer = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(store))
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)

		ch, err = store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.Nil(t, ch)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not empty with jwt and clientId", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Developer = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(store))
		fullBody, err := createEditConfigBody("jwtBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)

		ch, err = store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.Nil(t, ch)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not empty with only userid", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Developer = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(store))
		fullBody, err := createEditConfigBody("userIdBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)

		ch, err = store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.Nil(t, ch)
		assert.NoError(t, err)
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(bad.New()))
		fullBody, err := createEditConfigBody("fullBody")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})

	t.Run("should fail with no body", func(t *testing.T) {
		out, err := a.DeleteConfigurationV2(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingBody, err)
	})

	t.Run("should fail if jwt with no corresponding clientId", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(nil))
		fullBody, err := createEditConfigBody("missingClientId")
		assert.Nil(t, err)
		ctx = params.SetBody(ctx, fullBody)
		out, err := a.DeleteConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingClientID, err)
	})

}
