package api

import (
	"context"
	"testing"

	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/data/model"
	"code.justin.tv/extensions/fulton-configuration/data/model/bad"
	"code.justin.tv/extensions/fulton-configuration/data/model/memory"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestSetChannelConfiguration(t *testing.T) {
	a := &API{}
	t.Run("should report missing body", func(t *testing.T) {
		out, err := a.SetChannelConfiguration(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, params.ErrMissingBody, err)
	})

	t.Run("should report unimplemented without manager", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Broadcaster: protocol.NewRecord("2", nil),
		})
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should report illegal segment channel", func(t *testing.T) {
		t.Run("for broadcaster", func(t *testing.T) {
			ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
				ExtensionID: "extID",
				Broadcaster: protocol.NewRecord("2", nil),
			})
			ctx = data.Store(ctx, createManager(nil))
			out, err := a.SetChannelConfiguration(ctx)
			assert.Nil(t, out)
			assert.Equal(t, protocol.ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
		})

		t.Run("for developer", func(t *testing.T) {
			ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
				ExtensionID: "extID",
				Developer:   protocol.NewRecord("2", nil),
			})
			ctx = data.Store(ctx, createManager(nil))
			out, err := a.SetChannelConfiguration(ctx)
			assert.Nil(t, out)
			assert.Equal(t, protocol.ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
		})
	})

	t.Run("should report missing extension ID", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			Broadcaster: protocol.NewRecord("2", nil),
		})
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingExtensionIDCode, errors.GetErrorCode(err))
	})

	t.Run("should require authentication", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Broadcaster: protocol.NewRecord("2", nil),
		})
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnauthorized, err)
	})

	t.Run("should report illegal length", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Broadcaster: protocol.NewRecord("2", makeLongString(protocol.MaxSegmentLength+1)),
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrIllegalSegmentLengthCode, errors.GetErrorCode(err))
	})

	t.Run("should succeed if created", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Broadcaster: protocol.NewRecord("2", nil),
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(store))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
		ch, err := store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.NoError(t, err)
		require.NotNil(t, ch)
		assert.NotNil(t, ch.Broadcaster)
	})

	t.Run("should succeed if updating", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Developer = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Developer:   protocol.NewRecord("2", nil),
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(store))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
		ch, err = store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.NoError(t, err)
		require.NotNil(t, ch)
		require.NotNil(t, ch.Developer)
		assert.Equal(t, "2", ch.Developer.Version())
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetChannelConfigurationInput{
			ExtensionID: "extID",
			ChannelID:   "chID",
			Broadcaster: protocol.NewRecord("2", nil),
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})
}

func makeLongString(n int) *string {
	b := make([]rune, n)
	s := string(b)
	return &s

}
