package api

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/protocol"
	"code.justin.tv/gds/gds/golibs/params"
	"context"
)

func (a *API) SetChannelConfigurationV2(ctx context.Context) (interface{}, error) {
	p := params.New(ctx)
	reporter := telemetry.SampleReporterWithContext(a.Reporter, ctx)

	var in protocol.SetChannelConfigurationInputV2
	if err := p.ParseJSONBody(&in); err != nil {
		convertAndReportResponse(ctx, "set_channel_configuration_v2", err, reporter)
		return nil, err
	}
	creds := auth.Load(ctx)

	barbradyParams := map[string]string{}
	if in.ExtJwt != "" {
		barbradyParams[auth.ParamJWT] = in.ExtJwt
		if in.ClientId == "" {
			convertAndReportResponse(ctx, "set_channel_configuration_v2", protocol.ErrMissingClientID, reporter)
			return nil, protocol.ErrMissingClientID
		}
		barbradyParams[auth.ParamClientID] = in.ClientId
	}

	if in.UserID != "" {
		barbradyParams[auth.ParamUserID] = in.UserID
	}

	ext, err := data.Load(ctx).ForExtension(in.ExtensionID)
	if err != nil {
		convertAndReportResponse(ctx, "set_channel_configuration_v2", err, reporter)
		return nil, err
	}
	if in.Developer != nil {
		seg, err := protocol.Developer(in.ChannelID)
		if err != nil {
			convertAndReportResponse(ctx, "set_channel_configuration_v2", err, reporter)
			return nil, err
		}

		if barbradyCreds, ok := creds.(*auth.BarbradyCredentials); ok {
			barbradyParams[auth.ParamExtensionID] = in.ExtensionID
			barbradyCreds.RequestCapabilities(
				[]string{auth.CapEditDeveloperConfig},
				barbradyParams,
			)
		}
		ext.Set(seg, in.Developer)
	}
	if in.Broadcaster != nil {
		seg, err := protocol.Broadcaster(in.ChannelID)
		if err != nil {
			convertAndReportResponse(ctx, "set_channel_configuration_v2", err, reporter)
			return nil, err
		}

		if barbradyCreds, ok := creds.(*auth.BarbradyCredentials); ok {
			barbradyParams[auth.ParamChannelID] = in.ChannelID
			barbradyCreds.RequestCapabilities(
				[]string{auth.CapEditBroadcasterConfig},
				barbradyParams,
			)
		}
		ext.Set(seg, in.Broadcaster)
	}

	err = ext.Save()
	convertAndReportResponse(ctx, "set_channel_configuration_v2", err, reporter)
	return nil, err
}
