package configuration

import (
	"code.justin.tv/extensions/fulton-configuration/auth"
	"code.justin.tv/extensions/fulton-configuration/data"
	"code.justin.tv/extensions/fulton-configuration/server/api"
	"code.justin.tv/extensions/instrumentation"
	htmid "code.justin.tv/gds/gds/golibs/http"
	hmid "code.justin.tv/gds/gds/golibs/hystrix"
	"code.justin.tv/infosec/cors"
	"github.com/afex/hystrix-go/hystrix"
	goji "goji.io"
	"goji.io/pat"
	"log"
	"net/http"
)

var (
	corsPolicy = cors.Policy{
		AllowedOrigins: cors.Origins("*"),
		AllowMethods:   cors.Methods("GET", "POST", "PUT", "DELETE"),
		AllowHeaders:   cors.Headers("Accept", "Authorization", "Client-ID", "Twitch-Api-Token", "X-Forwarded-Proto", "X-Requested-With", "X-CSRF-Token", "Content-Type"),
	}
)

// BuildServer combines all parts that make up smart to create functioning handlers
func BuildServer(hystrixTimeout int, handler auth.Handler, mgr data.ConfigurationManager, a *api.API) *goji.Mux {
	server := goji.NewMux()
	server.Use(corsPolicy.MustMiddleware)
	server.Use(cors.BlockOnOptions)
	server.Use(data.StoreErrorConverter())
	server.Use(hmid.CreateCircuit("unknown"))
	server.Use(auth.StoreHandler(handler))
	server.Use(data.StoreManager(mgr))

	hystrix.DefaultTimeout = hystrixTimeout
	hystrix.DefaultMaxConcurrent = 2000
	hystrix.DefaultSleepWindow = 1000
	hystrix.DefaultErrorPercentThreshold = 60

	AppendFunctions(server, a)

	return server
}

func AppendFunctions(mux *goji.Mux, a *api.API) {
	mux.HandleFunc(pat.Get("/ping"), func(w http.ResponseWriter, r *http.Request) {
		_, err := w.Write([]byte("OK"))
		if err != nil {
			log.Println("Unable to write 'OK' to response for a /ping request")
		}
	})
	mux.Handle(pat.Get("/channel/:cid"), htmid.NewAPIHandler("get_channel_configuration", instrumentation.Frequent, a.GetChannelConfiguration))
	mux.Handle(pat.Put("/channel/:cid"), htmid.NewAPIHandler("set_channel_configuration", instrumentation.Common, a.SetChannelConfiguration))
	mux.Handle(pat.Delete("/channel/:cid"), htmid.NewAPIHandler("delete_channel_configuration", instrumentation.Rare, a.DeleteConfiguration))
	mux.Handle(pat.Post("/addresses"), htmid.NewAPIHandler("get_configuration", instrumentation.Common, a.GetConfiguration))
	mux.Handle(pat.Put("/addresses"), htmid.NewAPIHandler("set_configuration", instrumentation.Common, a.SetConfiguration))

	// V2  Routes
	mux.Handle(pat.Delete("/channel/:cid/v2"), htmid.NewAPIHandler("delete_channel_configuration", instrumentation.Rare, a.DeleteConfigurationV2))
	mux.Handle(pat.Put("/channel/:cid/v2"), htmid.NewAPIHandler("set_channel_configuration", instrumentation.Common, a.SetChannelConfigurationV2))
	mux.Handle(pat.Put("/addresses/v2"), htmid.NewAPIHandler("set_configuration", instrumentation.Common, a.SetConfigurationV2))
}
