package storage

import (
	"context"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	dynamoDBTimeout = 5 * time.Second
)

// PricingDao used to access or persist pricing data
type PricingDao struct {
	ddb dynamodbiface.DynamoDBAPI
}

// PricingEntry represents an entry in the pricing database
type PricingEntry struct {
	Vegetable string `json:"Vegetable"`
	Price     int32  `json:"Price"`
}

// NewPricingDao constructs a new PricingDao
func NewPricingDao(ddb dynamodbiface.DynamoDBAPI) *PricingDao {
	return &PricingDao{ddb}
}

// Put updates or records a new price
func (p *PricingDao) Put(ctx context.Context, entry *PricingEntry) (*PricingEntry, error) {
	av, err := dynamodbattribute.MarshalMap(entry)

	if err != nil {
		return nil, err
	}

	input := &dynamodb.PutItemInput{
		Item:      av,
		TableName: aws.String(os.Getenv("PRICING_TABLE_NAME")),
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()
	output, err := p.ddb.PutItemWithContext(ctx, input)

	if err != nil {
		return nil, err
	}

	pricing := PricingEntry{}
	err = dynamodbattribute.UnmarshalMap(output.Attributes, &pricing)
	if err != nil {
		return nil, err
	}

	return &pricing, nil
}

// Get retrieves the current price for a vegetable by name
func (p *PricingDao) Get(ctx context.Context, vegetable string) (*PricingEntry, error) {
	input := &dynamodb.GetItemInput{
		TableName:      aws.String(os.Getenv("PRICING_TABLE_NAME")),
		ConsistentRead: aws.Bool(true),
		Key: map[string]*dynamodb.AttributeValue{
			"Vegetable": {
				S: aws.String(vegetable),
			},
		},
	}

	ctx, cancel := context.WithDeadline(ctx, time.Now().Add(dynamoDBTimeout))
	defer cancel()
	result, err := p.ddb.GetItemWithContext(ctx, input)

	if err != nil {
		return nil, err
	}

	entry := PricingEntry{}

	err = dynamodbattribute.UnmarshalMap(result.Item, &entry)

	if err != nil {
		return nil, err
	}

	return &entry, nil
}
