package storage_test

import (
	"code.justin.tv/common/go_test_dynamo"
	"context"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/stretchr/testify/assert"
	"code.justin.tv/extensions/fulton-configuration/storage"
	"os"
	"testing"
)

const (
	testCarrotName   = "Carrot"
	testCarrotPrice  = 199
	testVegetableKey = "Vegetable"
)

// TestGetOnEmptyTable validates the Get response when the table is empty (and requested item does not exist in the table)
func TestGetOnEmptyTable(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	pricing := constructPricingTestDao(t)

	// Call Get without any entries in the table and validate the response
	actualEntry, err := pricing.Get(context.Background(), testCarrotName)
	assert.Nil(t, err, "Get should not return errors")

	expectedEntry := storage.PricingEntry{}
	assert.NotNil(t, actualEntry, "Get should return a non-nil PricingEntry")
	assert.Equal(t, expectedEntry.Vegetable, actualEntry.Vegetable, "Entry's Vegetable does not match expected")
	assert.Equal(t, expectedEntry.Price, actualEntry.Price, "Entry's Price does not match expected")
}

// TestGetOnValidEntry validates the Get response when the table has the desired entry
func TestGetOnValidEntry(t *testing.T) {
	defer go_test_dynamo.Instance().Cleanup()
	pricing := constructPricingTestDao(t)

	// Add an entry to the table
	expectedGetEntry := storage.PricingEntry{
		Vegetable: testCarrotName,
		Price:     testCarrotPrice,
	}
	returnedPutEntry, err := pricing.Put(context.Background(), &expectedGetEntry)

	assert.Nil(t, err, "Put should not return errors")
	// By default, DynamoDB's Put will not return the newly Put entry
	expectedPutEntry := storage.PricingEntry{}
	assert.NotNil(t, returnedPutEntry, "Put should return a non-nil PricingEntry")
	assert.Equal(t, expectedPutEntry.Vegetable, returnedPutEntry.Vegetable, "Entry's Vegetable does not match expected")
	assert.Equal(t, expectedPutEntry.Price, returnedPutEntry.Price, "Entry's Price does not match expected")

	// Now, call Get on the entry and validate the response
	returnedGetEntry, err := pricing.Get(context.Background(), testCarrotName)
	assert.Nil(t, err, "Get should not return errors")
	assert.NotNil(t, returnedGetEntry, "Get should return a non-nil PricingEntry")
	assert.Equal(t, expectedGetEntry.Vegetable, returnedGetEntry.Vegetable, "Entry's Vegetable does not match expected")
	assert.Equal(t, expectedGetEntry.Price, returnedGetEntry.Price, "Entry's Price does not match expected")

}

// Helper method to construct a Pricing DAO using go_test_dynamo
func constructPricingTestDao(t *testing.T) *storage.PricingDao {
	// Set the table name
	os.Setenv("PRICING_TABLE_NAME", "Test_Dynamo_Vegetables")

	// Get the Dynamo client and construct a table
	dbapi := go_test_dynamo.Instance().Dynamo
	err := constructVeggieDynamoPricingTable(dbapi)
	assert.Nil(t, err, "Expected DynamoDB table to be constructed without errors")

	// Construct our PricingDAO using the go_test_dynamo client
	return storage.NewPricingDao(dbapi)
}

// Helper method to construct a new Vegetable Pricing DynamoDB table
// See: https://docs.aws.amazon.com/sdk-for-go/v1/developer-guide/dynamo-example-create-table.html
func constructVeggieDynamoPricingTable(ddb dynamodbiface.DynamoDBAPI) error {
	input := &dynamodb.CreateTableInput{
		AttributeDefinitions: []*dynamodb.AttributeDefinition{
			{
				AttributeName: aws.String(testVegetableKey),
				AttributeType: aws.String("S"),
			},
		},
		KeySchema: []*dynamodb.KeySchemaElement{
			{
				AttributeName: aws.String(testVegetableKey),
				KeyType:       aws.String("HASH"),
			},
		},
		ProvisionedThroughput: &dynamodb.ProvisionedThroughput{
			ReadCapacityUnits:  aws.Int64(5),
			WriteCapacityUnits: aws.Int64(5),
		},
		TableName: aws.String(os.Getenv("PRICING_TABLE_NAME")),
	}

	_, err := ddb.CreateTableWithContext(context.Background(), input)
	return err
}
