#!/usr/bin/env python
import boto3
import glob
import re
import json
import sys
import os
import subprocess
from shutil import copyfile, rmtree
from cryptography.hazmat.primitives import serialization as crypto_serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.hazmat.backends import default_backend as crypto_default_backend


def clean(s3_bucket):
    try:
        subprocess.call("terraform destroy -force", shell=True)
    except Exception as e:
        print("type error: " + str(e))

    # kill local resources
    for folder in ["kubernetes", ".terraform", "generated"]:
        rmtree(folder, True)
    # kill state bucket will all versions
    delete_s3_bucket(s3_bucket)
    # kill remaining local resources
    for file in ["backend.tf"]:
        os.remove(file)
    for ext in ['.pem', '.pub', '.out']:
        files = glob.glob("./*" + ext)
        for file in files:
            os.remove(file)
    write_terraform_vars([''], "", "", "", "")
    return


def create_ssh_keypair(cluster_name):
    key_name = cluster_name + "_node_key"
    key = rsa.generate_private_key(
        backend=crypto_default_backend(),
        public_exponent=65537,
        key_size=4096
    )
    private_key = key.private_bytes(
        crypto_serialization.Encoding.PEM,
        crypto_serialization.PrivateFormat.PKCS8,
        crypto_serialization.NoEncryption())
    public_key = key.public_key().public_bytes(
        crypto_serialization.Encoding.OpenSSH,
        crypto_serialization.PublicFormat.OpenSSH
    )
    # save keypair
    if os.path.exists(key_name + ".pem") or os.path.exists(key_name + ".pub"):
        print("[WARN] KeyFiles Already Exist! Not Overwriting")
    else:
        print(private_key.decode('utf-8'), file=open(key_name + ".pem", "w"))
        print(public_key.decode('utf-8'), file=open(key_name + ".pub", "w"))
    return key_name


def create_s3_bucket(s3_bucket, region):
    # connect to s3 and create bucket
    s3 = boto3.client('s3')
    try:
        s3.create_bucket(
            Bucket=s3_bucket,
            ACL='private',
            CreateBucketConfiguration={
                'LocationConstraint': region
            }
        )
    except s3.exceptions.ClientError as e:
        print("[WARN] " + str(e))

    # enable versioning on bucket
    s3.put_bucket_versioning(
        Bucket=s3_bucket,
        VersioningConfiguration={
            'Status': 'Enabled'
        }
    )

    # output name of bucket
    print("[INFO] Bucket Name: {}".format(s3_bucket))
    return s3_bucket


def delete_s3_bucket(s3_bucket):
    s3 = boto3.client('s3')
    response = s3.list_objects_v2(Bucket=s3_bucket)
    if 'Contents' in response:
        for item in response['Contents']:
            print('deleting file', item['Key'])
            s3.delete_object(Bucket=s3_bucket, Key=item['Key'])
            while response['KeyCount'] == 1000:
                response = s3.list_objects_v2(
                    Bucket=s3_bucket,
                    StartAfter=response['Contents'][0]['Key'],
                )
                for item in response['Contents']:
                    print('deleting file', item['Key'])
                    s3.delete_object(Bucket=s3_bucket, Key=item['Key'])
    s3.put_bucket_versioning(
        Bucket=s3_bucket,
        VersioningConfiguration={
            'Status': 'Suspended'
        }
    )
    subprocess.call("sh ../bin/s3_delete.sh " + s3_bucket, shell=True)
    s3.delete_bucket(Bucket=s3_bucket)


def extract_bucket_name(file_name):
    with open(file_name, 'rt') as in_file:
        for line in in_file:
            if "bucket" in line:
                return re.findall('"([^"]*)"', line.strip())[0]


def ingest_config(filename):
    with open(filename, 'r') as F:
        config = json.loads(F.read() or '{}')

    config['route53_domain'] = config.get('cluster_env') + '.' + config.get('cluster_domain')

    # convert FQDN to string - dot to dash
    s3_bucket = "{0}-state-store".format(re.sub("[\W\d]+", "-", config.get('route53_domain').lower().strip()))
    config['bucket_name'] = s3_bucket

    return config


def init_terraform():
    os.system("terraform init")
    os.system("terraform plan -out plan.out")
    os.system("terraform apply plan.out")


def install_sshkey(key_name):
    pem_file = key_name + ".pem"
    user_ssh_path = os.path.expanduser('~/.ssh/')
    try:
        copyfile(pem_file, user_ssh_path + pem_file)
        # pem files need restrictive attributes
        os.chmod((user_ssh_path + pem_file), 0o600)
        print("[INFO] " + pem_file + " has been installed to " + user_ssh_path)
    except OSError as e:
        print("[WARN]" + str(e))
        print("[WARN] Unable to copy " + pem_file + " to " + user_ssh_path + "!\n Try Manually with CHMOD 0600")


def upload_sshkey(keyname, region):
    subprocess.call(['aws', 'ec2', 'import-key-pair',
                     '--key-name', keyname,
                     '--region', region,
                     '--public-key-material', 'file://./' + keyname + '.pub'])


def validate_route53(domain_name):
    route53 = boto3.client('route53')
    r53_zones = route53.list_hosted_zones()
    for n in r53_zones.get('HostedZones'):
        if domain_name in n.values():
            print("[INFO] SUCCESS! Zone: " + domain_name + " exists in this account's Route53")
            return
    print("[ERROR] Zone: " + domain_name + " does not exist in this account's Route53!!")
    quit()


def write_terraform_vars(azs, cluster_region, cluster_env, cluster_domain, aws_account_id, key_name):
    tf_main_file = 'main.tf'
    subprocess.call(['sed', '-i',
                     '-e', 's/allowed_account_ids =.*/allowed_account_ids = ["%s"]/' % (aws_account_id,),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/cluster_name =.*/cluster_name = "%s"/' % (cluster_region,),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/cluster_domain =.*/cluster_domain = "%s"/' % (cluster_domain,),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/cluster_region =.*/cluster_region = "%s"/' % (cluster_region,),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/environment_name =.*/environment_name = "%s"/' % (cluster_env,),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/cluster_azs =.*/cluster_azs = %s/' % (str(azs).replace("'", "\""),),
                     tf_main_file])
    subprocess.call(['sed', '-i',
                     '-e', 's/key_name =.*/key_name = "%s"/' % (key_name,),
                     tf_main_file])
    print(

    )


def write_terraform_backend(s3_bucket, key_name, region):
    # probably best to not mess with this mess >
    print(
        "// **DO NOT MODIFY** \n"
        "// This File is Managed by init.py\n"
        "\n"
        "terraform {\n"
        "\tbackend \"s3\" {\n"
        "\t\tbucket = \"" + s3_bucket + "\"\n"
        "\t\tkey    = \"" + key_name + "\"\n"
        "\t\tregion = \"" + region + "\"\n"
        "\t}\n"
        "}\n", file=open("backend.tf", "w+")
    )


if len(sys.argv) > 1:
    args = sys.argv[1:]
    if "nukem" in args:
        config_in = ingest_config(sys.argv[1])
        bucket_name = extract_bucket_name("backend.tf")
        clean(bucket_name)
    else:
        nargs = len(sys.argv)

        # read in config file, store params
        config_in = ingest_config(sys.argv[1])

        # validate Route53 Zone exists
        validate_route53(config_in['route53_domain'] + '.')

        # create state store bucket
        create_s3_bucket(config_in['bucket_name'], config_in['cluster_region'])

        # create ssh keypair and install on host
        ssh_key_name = create_ssh_keypair(config_in['route53_domain'])
        upload_sshkey(ssh_key_name, config_in['cluster_region'])
        install_sshkey(ssh_key_name)

        # append ".tfstate" to cluster_name for tf key
        tf_backend_key = config_in['bucket_name'] + ".tfstate"

        # write out backend and vars info for Terraform state bucket
        write_terraform_backend(config_in['bucket_name'], tf_backend_key, config_in['cluster_region'])
        write_terraform_vars(config_in['availability_zones'], config_in['cluster_region'],
                             config_in['cluster_env'], config_in['route53_domain'], config_in['aws_account_id'],
                             ssh_key_name)

        # initialize terraform
        init_terraform()
else:
    print("[ERROR] Requires config file location as first parameter")
