package main

import (
	"crypto/tls"
	"fmt"
	"log"
	"net/http"
	"os"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/orchestration/fultonlibs/FultonGoLangBootstrap/bootstrap"
	"code.justin.tv/extensions/orchestration/fultonlibs/FultonGoLangBootstrap/fultonecs"
	"code.justin.tv/extensions/orchestration/service/api"
	"code.justin.tv/extensions/orchestration/service/config"
	"code.justin.tv/extensions/orchestration/service/routes"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/security/ephemeralcert"
	"github.com/heroku/rollbar"

	validator "gopkg.in/validator.v2"
)

var (
	serviceName = "orchestrator"
)

//
// This function adds environment variables that the Fulton bootstrap process requires.
//
// Environment variables that are provided by ECS when it starts the containers up, as we have defined in
// the /buildspec.yml file:
//
//  ENVIRONMENT
//  GIT_COMMIT
//
// Env vars required by the Fulton bootstrap process:
//
//  FULTON_STAGE - matches the ENVIRONMENT var we currently use
//  FULTON_SUBSTAGE - not important, has a default value "primary" if unset
//  AWS_DEFAULT_REGION - region env var provided by ECS
//  ECS_CLUSTER - the name of the cluster running this app, used as a dimension in cloudwatch to identify which cluster
//     originated which metrics
//  HOSTNAME - not important when running in ECS
//  FULTON_VERSION - this can match GIT_COMMIT
//
// AWS_DEFAULT_REGION is required by the cloudwatch sender
// AWS_DEFAULT_REGION and FULTON_STAGE are required by c7 loader to identify the most relevant config params
//
func createFultonEnvironmentVars() {
	env := os.Getenv("ENVIRONMENT")

	if env == "" {
		env = "local"
		os.Setenv("ENVIRONMENT", env)
	}

	// temporary code to find out what env vars are available to the running container in an actual ECS environment
	if env != "local" {
		for _, envName := range os.Environ() {
			fmt.Println(envName)
		}
	}

	gitCommit := os.Getenv("GIT_COMMIT")

	if gitCommit == "" {
		gitCommit = "unknown"
		os.Setenv("GIT_COMMIT", gitCommit)
	}

	os.Setenv("FULTON_STAGE", env)
	os.Setenv("FULTON_VERSION", gitCommit)

	// ECS_CLUSTER just needs to identify the cluster that the metrics were emitted from
	os.Setenv("ECS_CLUSTER", serviceName)
}

func listenAndServeTLS(h http.Handler, serverConfig *twitchserver.ServerConfig, tlsConfig *tls.Config) error {
	ln, err := tls.Listen("tcp", serverConfig.Addr, tlsConfig)
	if err != nil {
		return err
	}
	return twitchserver.Serve(ln, h, serverConfig)
}

func main() {
	createFultonEnvironmentVars()

	env := os.Getenv("ENVIRONMENT")
	gitCommit := os.Getenv("GIT_COMMIT")

	fmt.Printf("Starting server from GIT commit %s in %s\n", gitCommit, env)

	// This loads the config from "all.c7"
	bs, err := fultonecs.ECS(serviceName, bootstrap.DefaultConfigSetLoader)
	if err != nil {
		panic(fmt.Sprintf("Error bootstrapping: %s", err))
	}

	bs.SampleReporter.Report("ServerColdStart", 1, telemetry.UnitCount)

	conf := config.Config{}

	err = bs.C7.FillWithNamespace(serviceName, &conf)
	if err != nil {
		panic(fmt.Sprintf("Error loading config: %s", err))
	}

	err = validator.Validate(conf)

	if err != nil {
		panic(fmt.Sprintf("Error in configuration: %s", err))
	}

	apiHandler, err := api.NewAPI()
	if err != nil {
		log.Fatalf("Couldn't create the API")
	}
	twitchserver.AddDefaultSignalHandlers()
	server := routes.BuildServer(apiHandler, &conf, &bs.SampleReporter)

	location := fmt.Sprintf(":%d", conf.Port)
	httpLocation := fmt.Sprintf(":%d", conf.HTTPPort)
	httpDebugLocation := fmt.Sprintf(":%d", conf.HTTPDebugPort)

	bs.Logger.Log("Server listening on "+location,
		"ProcessAddressLaunchID", bs.ProcessIdentifier.LaunchID,
		"ProcessAddressMachine", bs.ProcessIdentifier.Machine,
		"ProcessAddressVersion", bs.ProcessIdentifier.Version,
		"ServiceTupleStage", bs.ProcessIdentifier.Stage,
		"ServiceTupleSubstage", bs.ProcessIdentifier.Substage,
		"ServiceTupleService", bs.ProcessIdentifier.Service,
		"ServiceTupleRegion", bs.ProcessIdentifier.Region)
	go func() {
		err = twitchserver.ListenAndServe(twitchserver.NewServer(), &twitchserver.ServerConfig{
			Addr:      httpLocation,
			DebugAddr: httpDebugLocation,
		})
		if err != nil {
			err = fmt.Errorf("ListenAndServe (healthcheck) failed with: %s", err)
			rollbar.Error(rollbar.ERR, err)
			log.Print(err)
		}
	}()
	tlsConfig, err := ephemeralcert.GenerateTLSConfig()
	// select RSA Ciphers to avoid gigantic TLS overhead which can be so bad that it trips Hystrix
	tlsConfig.CipherSuites = []uint16{
		tls.TLS_RSA_WITH_RC4_128_SHA,
		tls.TLS_RSA_WITH_3DES_EDE_CBC_SHA,
		tls.TLS_RSA_WITH_AES_128_CBC_SHA,
		tls.TLS_RSA_WITH_AES_256_CBC_SHA,
		tls.TLS_RSA_WITH_AES_128_CBC_SHA256,
		tls.TLS_RSA_WITH_AES_128_GCM_SHA256,
		tls.TLS_RSA_WITH_AES_256_GCM_SHA384,
	}
	if err != nil {
		log.Fatalf("Could not generate TLS configuration: %v\n", err)
		rollbar.Error(rollbar.ERR, err)
	}
	err = listenAndServeTLS(server.Mux, &twitchserver.ServerConfig{
		Addr: location,
	}, tlsConfig)
	if err != nil {
		err = fmt.Errorf("ListenAndServeTLS failed with: %s", err)
		rollbar.Error(rollbar.ERR, err)
		log.Print(err)
	}
}
