package fultonecs

import (
	"encoding/json"
	"errors"
	"os"
	"time"

	"code.justin.tv/amzn/TwitchLoggingCommonLoggers"
	"code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"

	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrap"
	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrapecs"
)

const (
	metricsFlushInterval     = 30 * time.Second
	metricsBufferSize        = 100000
	metricsAggregationPeriod = time.Minute // Use one-minute metrics.
)

// ECS understands how to create the FultonBootstrap object for the ECS platform
// serviceName: the name of the service you are running
func ECS(serviceName string, configSetLoader simplebootstrap.ConfigSetLoader) (*simplebootstrap.FultonBootstrap, error) {
	// Determine the process identifiers (required for default logging and metrics)
	tPid, err := simplebootstrapecs.GetECSProcessIdentifiers(serviceName)
	if err != nil {
		return nil, err
	}
	if tPid == nil {
		return nil, errors.New("cannot use empty process identifier")
	}

	// Construct the logger
	stdoutWriter := &telemetry.SampleReportingWriter{
		Writer:                 os.Stdout,
		BytesWrittenMetricName: telemetry.MetricStdoutBytesWritten,
		DurationMetricName:     telemetry.MetricStdoutWaitTime,
	}

	// Construct the logging OpMonitor
	// TODO: Pipe somewhere else?
	encoder := json.NewEncoder(stdoutWriter)
	// TODO: Do we want a fallback logger?
	jsonLogger := &loggers.JSONLogger{Dest: encoder, OnError: nil}

	// Construct the sample observer, requires tPid.Region and Service fields
	sender := cw.NewUnbuffered(tPid, jsonLogger)
	sampleObserver := telemetry.NewBufferedAggregator(metricsFlushInterval, metricsBufferSize, metricsAggregationPeriod, sender, jsonLogger)

	customMembers := simplebootstrap.FultonCustomMembers{
		Logger:           jsonLogger,
		SampleObserver:   sampleObserver,
		AutoFlushMetrics: false, // Batch (don't flush every time)
	}

	simpleBootstrap, err := simplebootstrapecs.ECS(serviceName, customMembers, configSetLoader)
	if err != nil {
		return nil, err
	}
	if simpleBootstrap == nil {
		return nil, errors.New("cannot use empty bootstrap")
	}

	// This is a pretty hacky way of getting logging to properly output metrics
	// TODO: Fix the circular dependency
	stdoutWriter.Reporter = &simpleBootstrap.SampleReporter
	return simpleBootstrap, nil

}
