package discoveryapi_test

import (
	"context"
	"errors"
	"testing"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc/discoveryfakes"
	"code.justin.tv/extensions/orchestration/service/api/discoveryapi"
	"code.justin.tv/extensions/orchestration/service/clients/discoman"
	"code.justin.tv/gds/gds/golibs/params"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDeleteExtensionVersionDiscoveryDataAPICall(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	eid, ver := "34567", "3.4.5"
	ctx = params.Set(ctx, "id", eid)
	ctx = params.Set(ctx, "version", ver)

	stubResponse := &discovery.DeleteExtensionVersionDiscoveryDataResponse{
		Success: true,
	}
	stubError := errors.New("fake error")

	var captured struct {
		ctx context.Context
		eid string
		ver string
	}
	stub := func(ctx context.Context, extensionID, extensionVersion string) (*discovery.DeleteExtensionVersionDiscoveryDataResponse, error) {
		captured.ctx = ctx
		captured.eid = extensionID
		captured.ver = extensionVersion
		return stubResponse, stubError
	}
	apiCall := discoveryapi.DeleteExtensionVersionDiscoveryDataAPICall(stub)

	resp, err := apiCall(ctx)

	t.Run("should call the underlying func with the correct parameters", func(t *testing.T) {

		assert.Equal(t, ctxValue, captured.ctx.Value(ctxKey))
		assert.Equal(t, eid, captured.eid)
		assert.Equal(t, ver, captured.ver)
	})

	t.Run("should return the correct responses", func(t *testing.T) {

		assert.Exactly(t, stubResponse, resp.(*discovery.DeleteExtensionVersionDiscoveryDataResponse))
		assert.Exactly(t, stubError, err)
	})
}

func TestDeleteExtensionVersionDiscoveryData(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	fakeDiscovery := new(discoveryfakes.FakeDiscovery)
	ctx = discoman.WithClient(ctx, fakeDiscovery)

	eid, ver := "78901", "0.1.2"

	t.Run("Success", func(t *testing.T) {

		resp := &discovery.DeleteExtensionVersionDiscoveryDataResponse{
			Success: true,
		}
		fakeDiscovery.DeleteExtensionVersionDiscoveryDataReturns(resp, nil)

		result, err := discoveryapi.DeleteExtensionVersionDiscoveryData(ctx, eid, ver)
		require.NoError(t, err)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.DeleteExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			assert.Equal(t, eid, actualReq.ExtensionID)
			assert.Equal(t, ver, actualReq.Version)
		})

		t.Run("should return the correct response", func(t *testing.T) {

			assert.True(t, result.Success)
		})
	})

	t.Run("Failure", func(t *testing.T) {

		stubErr := errors.New("fake error returned from discovery service")
		fakeDiscovery.DeleteExtensionVersionDiscoveryDataReturns(nil, stubErr)

		_, err := discoveryapi.DeleteExtensionVersionDiscoveryData(ctx, eid, ver)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.DeleteExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			assert.Equal(t, eid, actualReq.ExtensionID)
			assert.Equal(t, ver, actualReq.Version)
		})

		t.Run("should return the error", func(t *testing.T) {

			assert.Exactly(t, stubErr, err)
		})
	})
}
