package discoveryapi

import (
	"context"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/orchestration/service/api"
	"code.justin.tv/extensions/orchestration/service/auth"
	"code.justin.tv/extensions/orchestration/service/clients/discoman"
	"code.justin.tv/extensions/orchestration/service/clients/ems"
	"code.justin.tv/extensions/orchestration/service/config"
	"code.justin.tv/extensions/orchestration/service/protocol"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/golibs/params"
)

const (
	GetCategoriesParamType           = "type"
	GetCategoriesParamIncludeHidden  = "include_hidden"
	GetcategoriesParamIncludeDeleted = "include_deleted"
)

func GetCategories(ctx context.Context) (interface{}, error) {
	p := params.New(ctx)
	offset, limit, err := api.GetPagination(p)
	if err != nil {
		return nil, err
	}
	language := api.GetLanguage(p)
	catType := p.Get(GetCategoriesParamType)
	includeHidden := p.Get(GetCategoriesParamIncludeHidden) != ""
	includeDeleted := p.Get(GetcategoriesParamIncludeDeleted) != ""

	if config.RetrieveConfig(ctx).CallDiscovery {
		discoveryClient := discoman.RetrieveClient(ctx)

		req := &discovery.GetCategoriesRequest{
			Language:       language,
			Limit:          int32(limit),
			Offset:         int32(offset),
			IncludeDeleted: includeDeleted,
			IncludeHidden:  includeHidden,
			Type:           catType,
		}

		resp, err := discoveryClient.GetCategories(ctx, req)
		if err != nil {
			return nil, err
		}

		doc := protocol.FromDiscoveryCategories(resp.GetCategories())

		return doc, nil
	}

	emsClient := ems.RetrieveClient(ctx)
	resp, err := emsClient.GetCategories(ctx, documents.GetCategoriesRequest{
		Type:           catType,
		Language:       language,
		Offset:         offset,
		Limit:          limit,
		IncludeHidden:  includeHidden,
		IncludeDeleted: includeDeleted,
	}, &twitchclient.ReqOpts{
		AuthorizationToken: auth.Load(ctx),
	})

	if err != nil {
		return nil, err
	}

	return resp, nil
}
