package discoveryapi_test

import (
	"context"
	"errors"
	"testing"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc/discoveryfakes"
	"code.justin.tv/extensions/orchestration/service/api/discoveryapi"
	"code.justin.tv/extensions/orchestration/service/clients/discoman"
	"code.justin.tv/gds/gds/golibs/params"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetExtensionVersionDiscoveryDataAPICall(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	eid, ver := "23456", "2.3.4"
	ctx = params.Set(ctx, "id", eid)
	ctx = params.Set(ctx, "version", ver)

	stubResponse := &discovery.GetExtensionVersionDiscoveryDataResponse{
		ExtensionVersionData: &discovery.ExtensionVersionDiscoveryDocument{
			ExtensionID: eid,
			Version:     ver,
		},
	}
	stubError := errors.New("fake error")

	var captured struct {
		ctx context.Context
		eid string
		ver string
	}
	stub := func(ctx context.Context, extensionID, extensionVersion string) (*discovery.GetExtensionVersionDiscoveryDataResponse, error) {
		captured.ctx = ctx
		captured.eid = extensionID
		captured.ver = extensionVersion
		return stubResponse, stubError
	}
	apiCall := discoveryapi.GetExtensionVersionDiscoveryDataAPICall(stub)

	resp, err := apiCall(ctx)

	t.Run("should call the underlying func with the correct parameters", func(t *testing.T) {

		assert.Equal(t, ctxValue, captured.ctx.Value(ctxKey))
		assert.Equal(t, eid, captured.eid)
		assert.Equal(t, ver, captured.ver)
	})

	t.Run("should return the correct responses", func(t *testing.T) {

		assert.Exactly(t, stubResponse, resp.(*discovery.GetExtensionVersionDiscoveryDataResponse))
		assert.Exactly(t, stubError, err)
	})
}

func TestGetExtensionVersionDiscoveryData(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	fakeDiscovery := new(discoveryfakes.FakeDiscovery)
	ctx = discoman.WithClient(ctx, fakeDiscovery)

	eid := "78901"
	ver := "0.1.2"
	name := "some discovered extension"

	t.Run("Success", func(t *testing.T) {

		resp := &discovery.GetExtensionVersionDiscoveryDataResponse{
			ExtensionVersionData: &discovery.ExtensionVersionDiscoveryDocument{
				ExtensionID: eid,
				Version:     ver,
				DiscoveryData: &discovery.DiscoveryDataDocument{
					Name: name,
				},
			},
		}
		fakeDiscovery.GetExtensionVersionDiscoveryDataReturns(resp, nil)

		result, err := discoveryapi.GetExtensionVersionDiscoveryData(ctx, eid, ver)
		require.NoError(t, err)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.GetExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			assert.Equal(t, eid, actualReq.ExtensionID)
			assert.Equal(t, ver, actualReq.Version)
		})

		t.Run("should return the correct response", func(t *testing.T) {

			actual := result.ExtensionVersionData
			assert.Equal(t, eid, actual.ExtensionID)
			assert.Equal(t, ver, actual.Version)
			assert.Equal(t, name, actual.GetDiscoveryData().Name)
		})
	})

	t.Run("Failure", func(t *testing.T) {

		stubErr := errors.New("fake error returned from discovery service")
		fakeDiscovery.GetExtensionVersionDiscoveryDataReturns(nil, stubErr)

		_, err := discoveryapi.GetExtensionVersionDiscoveryData(ctx, eid, ver)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.GetExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			assert.Equal(t, eid, actualReq.ExtensionID)
			assert.Equal(t, ver, actualReq.Version)
		})

		t.Run("should return the error", func(t *testing.T) {

			assert.Exactly(t, stubErr, err)
		})
	})
}
