package discoveryapi_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io/ioutil"
	"strings"
	"testing"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc/discoveryfakes"
	"code.justin.tv/extensions/discovery/twirputils"
	"code.justin.tv/extensions/orchestration/service/api/discoveryapi"
	"code.justin.tv/extensions/orchestration/service/client"
	"code.justin.tv/extensions/orchestration/service/clients/discoman"
	"code.justin.tv/gds/gds/golibs/params"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestUpdateExtensionVersionDiscoveryDataAPICall(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	eid, ver := "12345", "1.2.3"
	ctx = params.Set(ctx, "id", eid)
	ctx = params.Set(ctx, "version", ver)

	name := "test extension"
	manifest := client.ExtensionDiscoveryDataInput{
		Name: &name,
	}
	body, err := json.Marshal(manifest)
	require.NoError(t, err)
	ctx = params.SetBody(ctx, ioutil.NopCloser(bytes.NewReader(body)))

	stubResponse := &discovery.PutExtensionVersionDiscoveryDataResponse{
		ExtensionVersionData: &discovery.ExtensionVersionDiscoveryDocument{
			ExtensionID: eid,
			Version:     ver,
			DiscoveryData: &discovery.DiscoveryDataDocument{
				Name: name,
			},
		},
	}
	stubError := errors.New("fake error")

	var captured struct {
		ctx      context.Context
		manifest client.ExtensionDiscoveryDataInput
	}
	stub := func(ctx context.Context, extensionID, version string, manifest client.ExtensionDiscoveryDataInput) (*discovery.PutExtensionVersionDiscoveryDataResponse, error) {
		captured.ctx = ctx
		captured.manifest = manifest
		return stubResponse, stubError
	}
	apiCall := discoveryapi.UpdateExtensionVersionDiscoveryDataAPICall(stub)

	resp, err := apiCall(ctx)

	t.Run("should call the underlying func with the correct parameters", func(t *testing.T) {
		assert.Equal(t, ctxValue, captured.ctx.Value(ctxKey))
		assert.Equal(t, manifest, captured.manifest)
	})

	t.Run("should return the correct responses", func(t *testing.T) {
		assert.Exactly(t, stubResponse, resp.(*discovery.PutExtensionVersionDiscoveryDataResponse))
		assert.Exactly(t, stubError, err)
	})
}

func TestUpdateExtensionVersionDiscoveryData(t *testing.T) {

	ctxKey, ctxValue := new(byte), "value to ensure base context was propagated"
	ctx := context.WithValue(context.Background(), ctxKey, ctxValue)

	eid, ver := "23456", "2.3.4"
	ctx = params.Set(ctx, "id", eid)
	ctx = params.Set(ctx, "version", ver)

	t.Run("Success", func(t *testing.T) {

		fakeDiscovery := new(discoveryfakes.FakeDiscovery)
		ctx = discoman.WithClient(ctx, fakeDiscovery)

		name := "some discovered extension"
		author := "fakeauthor"
		req := client.ExtensionDiscoveryDataInput{
			Name:       &name,
			AuthorName: &author,
		}

		resp := &discovery.PutExtensionVersionDiscoveryDataResponse{
			ExtensionVersionData: &discovery.ExtensionVersionDiscoveryDocument{
				ExtensionID: eid,
				Version:     ver,
				DiscoveryData: &discovery.DiscoveryDataDocument{
					Name:       name,
					AuthorName: author,
				},
			},
		}
		fakeDiscovery.PutExtensionVersionDiscoveryDataReturns(resp, nil)

		result, err := discoveryapi.UpdateExtensionVersionDiscoveryData(ctx, eid, ver, req)
		require.NoError(t, err)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.PutExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			actual := *(*actualReq).UpdateData
			expected := discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: eid,
				Version:     ver,
				Name:        twirputils.WrapStringPointer(&name),
				AuthorName:  twirputils.WrapStringPointer(&author),
			}
			assert.Equal(t, expected, actual)
		})

		t.Run("should return the correct response", func(t *testing.T) {

			assert.Equal(t, *resp, *result)
		})
	})

	t.Run("Failure", func(t *testing.T) {

		fakeDiscovery := new(discoveryfakes.FakeDiscovery)
		ctx = discoman.WithClient(ctx, fakeDiscovery)

		name := "some other discovered extension"
		author := "fakeauthor2"
		req := client.ExtensionDiscoveryDataInput{
			Name:       &name,
			AuthorName: &author,
		}

		stubErr := errors.New("fake error returned from discovery service")
		fakeDiscovery.PutExtensionVersionDiscoveryDataReturns(nil, stubErr)

		ctx = params.SetBody(ctx, ioutil.NopCloser(strings.NewReader("{}")))

		_, err := discoveryapi.UpdateExtensionVersionDiscoveryData(ctx, eid, ver, req)

		t.Run("should call discovery service with the correct parameters", func(t *testing.T) {

			actualCtx, actualReq := fakeDiscovery.PutExtensionVersionDiscoveryDataArgsForCall(0)

			assert.Equal(t, ctxValue, actualCtx.Value(ctxKey))
			actual := *(*actualReq).UpdateData
			expected := discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: eid,
				Version:     ver,
				Name:        twirputils.WrapStringPointer(&name),
				AuthorName:  twirputils.WrapStringPointer(&author),
			}
			assert.Equal(t, expected, actual)
		})

		t.Run("should return the error", func(t *testing.T) {

			assert.Exactly(t, stubErr, err)
		})
	})
}
