package api

import (
	"context"
	"encoding/base64"
	"strings"

	"code.justin.tv/extensions/orchestration/service/protocol"
	"code.justin.tv/extensions/orchestration/service/protocol/response"

	"code.justin.tv/extensions/orchestration/service/auth"
	"code.justin.tv/extensions/orchestration/service/clients/ems"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/golibs/params"
)

// GetExtensionsByKeys - get a list of extensions from IDs
func GetExtensionsByKeys(ctx context.Context) (interface{}, error) {
	p := params.New(ctx)
	keysEncoded := p.Get("keys")

	if keysEncoded == "" {
		return nil, protocol.ErrMissingExtensionKeys
	}

	keys := []string{}
	for _, encodedKey := range strings.Split(keysEncoded, ",") {
		key, err := base64.RawURLEncoding.DecodeString(encodedKey)
		if err != nil {
			return "", err
		}

		keys = append(keys, string(key))
	}

	emsClient := ems.RetrieveClient(ctx)

	extensions, err := emsClient.GetExtensionsByKeys(ctx, keys, &twitchclient.ReqOpts{
		AuthorizationToken: auth.Load(ctx),
	})

	if err != nil {
		return nil, err
	}

	docs := []*response.ExtensionDocument{}

	for _, extension := range extensions {
		docs = append(docs, protocol.FromEMSExtensionDocument(extension))
	}

	return docs, nil
}
