package auth

import (
	"context"
	"net/http"
)

const (
	TwitchAuthorizationHeader = "Twitch-Authorization"
)

var (
	cartmanHeaderMarker = 42                 // this value is never used, the address is important
	cartmanHeaderKey    = &cartmanHeaderMarker // unique within this process, private to this package
)

// StoreCartmanToken builds a goji middleware that automatically places the
// Cartman token into the context of each request or early outs with an
// unauthorized error.
func StoreCartmanToken() func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			token := r.Header.Get(TwitchAuthorizationHeader)
			ctx := context.WithValue(r.Context(), cartmanHeaderKey, token)
			inner.ServeHTTP(w, r.WithContext(ctx))
		})
	}
}

// Load retrieves credentials from a request context.
func Load(ctx context.Context) string {
	if value, ok := ctx.Value(cartmanHeaderKey).(string); ok {
		return value
	}
	return ""
}
