package client

import (
	"context"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/ems/protocol"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "extensions"
	sendExtensionChatCap  = "send_extension_chat_message"
)

// Client is the client library to the gds extension orchestrator service
type Client interface {
	// GetExtensions returns the complete list of extensions available for installation.
	// It supports search, sorting, and pagination.
	GetExtensions(ctx context.Context, params documents.GetExtensionsParams, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionsDocument, error)

	// GetExtensionsByKeys returns a list of extensions by protocol.ExtensionKey in the same order
	// they are requested. Assuming all keys provided are in the valid protocol.ExtensionKey
	// format, either an extension record or nil will be returned in the same order as the
	// keys are provided.
	GetExtensionsByKeys(ctx context.Context, keys []string, reqOpts *twitchclient.ReqOpts) ([]*documents.ExtensionDocument, error)

	// GetExtensionByID returns information for the given extension
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to query
	GetExtensionByID(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionDocument, error)

	// GetReleasedExtensionByID returns information for the released version of a given extension
	// extensionID - the client ID for the desired extension
	GetReleasedExtensionByID(ctx context.Context, extensionID string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionDocument, error)

	// GetExtensionsByChannelID returns the list of extensions that are installed on the given channel, along with information
	// about the actual installations
	GetExtensionsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionsDocument, error)

	// GetMobileExtensionsByChannelID returns the list of mobile extensions that are installed on the given channel, along with information
	// about the actual installations
	GetMobileExtensionsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionsDocument, error)

	// InstallExtension installs the given extension on the given channel
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to query
	// channelID - the ID of the channel onto which the extension should be installed
	InstallExtension(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstallationStatusDocument, error)

	// InstallExtensionV@ installs the given extension on the given channel
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to query
	// channelID - the ID of the channel onto which the extension should be installed
	InstallExtensionV2(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionDocument, error)

	// ActivateExtensions applies a set of activation changes to extensions on the given channel. The extensions
	// must already be installed on the channel.
	// channelID - the ID of the channel for which the installed extensions should be activated/deactivated
	// config - structure containing information about the set of extension activation changes to apply
	ActivateExtensions(ctx context.Context, channelID string, config documents.ActivationConfigurationParams, reqOpts *twitchclient.ReqOpts) (*documents.ActivationsDocument, error)

	// SetInstallationConfiguration sets the given extension's installation configuration on the given channel.  In order
	// to activate an extension, this configuration string must match the one (if any) present in the extension manifest.
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to query
	// channelID - the ID of the channel for which the installed extension should be activated/deactivated
	// requiredConfigurationString - string that must match the one in the extension manifest in order to activate
	SetInstallationConfiguration(ctx context.Context, extensionID, version, channelID string, requiredConfigurationString string, reqOpts *twitchclient.ReqOpts) error

	// SetFeatureFlags sets the the broadcaster's choices for feature flags on an extension installation.
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to set
	// channelID - the ID of the channel for which the installed extension should be configured
	// flags - the subset of feature flags the broadcaster wishes to set
	SetFeatureFlags(ctx context.Context, extensionID, version, channelID string, flags documents.SetFeatureFlagsDocument, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionDocument, error)

	// UninstallExtension uninstalls the given extension from the given channel
	// extensionID - the client ID for the desired extension
	// version - the specific version of that extension to query
	// channelID - the ID of the channel from which the extension should be uninstalled
	UninstallExtension(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) error

	// AddExtension creates a new extension (or updates an existing extension in test)
	// manifest - A structure containing all the information related to the extension to add
	AddExtension(ctx context.Context, manifest documents.Manifest, reqOpts *twitchclient.ReqOpts) error

	// AddExtensionV2 creates a new extension (or updates an existing extension in test)
	// manifest - A structure containing all the information related to the extension to add
	AddExtensionV2(ctx context.Context, manifest protocol.ExtensionManifest, reqOpts *twitchclient.ReqOpts) (*protocol.ExtensionManifest, error)

	// DeleteExtensionVersion deletes the given extension version.  This is a soft delete; authorized
	// users can undelete the version by transitioning it back to "testing"
	DeleteExtensionVersion(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) error

	// DeleteExtension deletes all versions of the given extension.  This is a soft delete; authorized
	// users can undelete any version of the extension by transitioning it back to "testing"
	DeleteExtension(ctx context.Context, extensionID string, reqOpts *twitchclient.ReqOpts) error

	// TransitionExtensionState moves an extension through the various phases of its lifecycle
	// (e.g., test --> in review, in review --> approved, etc)
	// extensionId - the client ID for the desired extension
	// version - the specific version of that extension that should change state
	// state - the desired target state; must be one of :
	// 	  ("test", "review", "approved", "rejected", "live", "deprecated")
	TransitionExtensionState(ctx context.Context, extensionID, version, state string, reqOpts *twitchclient.ReqOpts) error

	// GetExtensionDiscovery gets the discovery metadata for an extension
	GetExtensionDiscovery(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) (*discovery.ExtensionVersionDiscoveryDocument, error)

	// UpdateExtensionDiscovery updates the discovery metadata for an extension
	// manifest - A structure containing all the information related to the discovery data of an extension
	UpdateExtensionDiscovery(ctx context.Context, extensionID, version string, input ExtensionDiscoveryDataInput, reqOpts *twitchclient.ReqOpts) (*discovery.ExtensionVersionDiscoveryDocument, error)

	// DeleteExtensionDiscovery deletes the discovery metadata for an extension
	DeleteExtensionDiscovery(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	twitchclient.Client
}

// NewClient creates an http client to call the EMS endpoints
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	return internalNewClient(conf)
}

func internalNewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	return &clientImpl{
		twitchClient,
	}, err
}
