package client

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/foundation/twitchclient"
)

func (c *clientImpl) GetExtensionDiscovery(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/developer/extensions/%s/%s/discovery", extensionID, version)}

	req, err := c.NewRequest(http.MethodGet, url.String(), nil)
	if err != nil {
		return nil, err
	}

	var out discovery.GetExtensionVersionDiscoveryDataResponse
	if err := c.execute(ctx, req, "discovery.get_extension_discovery", &out, reqOpts); err != nil {
		return nil, err
	}

	return out.GetExtensionVersionData(), nil
}

func (c *clientImpl) UpdateExtensionDiscovery(ctx context.Context, extensionID, version string, input ExtensionDiscoveryDataInput, reqOpts *twitchclient.ReqOpts) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/developer/extensions/%s/%s/discovery", extensionID, version)}

	bodyBytes, err := json.Marshal(input)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	var out discovery.PutExtensionVersionDiscoveryDataResponse
	if err := c.execute(ctx, req, "discovery.update_extension_discovery", &out, reqOpts); err != nil {
		return nil, err
	}

	return out.GetExtensionVersionData(), nil
}

func (c *clientImpl) DeleteExtensionDiscovery(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/developer/extensions/%s/%s/discovery", extensionID, version)}

	req, err := c.NewRequest(http.MethodDelete, url.String(), nil)
	if err != nil {
		return err
	}

	var out discovery.DeleteExtensionVersionDiscoveryDataResponse
	if err := c.execute(ctx, req, "discovery.delete_extension_discovery", &out, reqOpts); err != nil {
		return err
	}

	if !out.Success {
		return errors.New("delete extension unsuccessful")
	}

	return nil
}
