package client

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/ems/protocol"
)

func (c *clientImpl) GetExtensions(ctx context.Context, params documents.GetExtensionsParams, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionsDocument, error) {
	url := url.URL{Path: "/extensions/search"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_extensions",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionsDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetExtensionsByKeys(ctx context.Context, keys []string, reqOpts *twitchclient.ReqOpts) ([]*documents.ExtensionDocument, error) {
	url := url.URL{
		Path:     "/extensions",
		RawQuery: fmt.Sprintf("keys=%s", strings.Join(keys, ",")),
	}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_bulk_extensions_by_key",
		StatSampleRate: defaultStatSampleRate,
	})

	var document []*documents.ExtensionDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return document, nil
}

func (c *clientImpl) GetExtensionByID(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/extensions/%s/%s", extensionID, version)}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetReleasedExtensionByID(ctx context.Context, extensionID string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/extensions/%s", extensionID)}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_released_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetExtensionsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionsDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/v2", channelID)}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_extensions_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.InstalledExtensionsDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetMobileExtensionsByChannelID(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionsDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/mobile/channels/%s/extensions", channelID)}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_mobile_extensions_by_channel_id",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.InstalledExtensionsDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteExtensionVersion(ctx context.Context, extensionID, version string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/extensions/%s/%s", extensionID, version)}

	req, err := c.NewRequest("DELETE", url.String(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_extension_version",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) DeleteExtension(ctx context.Context, extensionID string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/extensions/%s", extensionID)}

	req, err := c.NewRequest("DELETE", url.String(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) InstallExtension(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstallationStatusDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/%s/%s", channelID, extensionID, version)}

	req, err := c.NewRequest("POST", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.install_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.InstallationStatusDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) InstallExtensionV2(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/%s/%s/v2", channelID, extensionID, version)}

	req, err := c.NewRequest("POST", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.install_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.InstalledExtensionDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) UninstallExtension(ctx context.Context, extensionID, version, channelID string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/%s/%s", channelID, extensionID, version)}

	req, err := c.NewRequest("DELETE", url.String(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.uninstall_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) AddExtension(ctx context.Context, manifest documents.Manifest, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: "/extensions"}

	bodyBytes, err := json.Marshal(manifest)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) AddExtensionV2(ctx context.Context, manifest protocol.ExtensionManifest, reqOpts *twitchclient.ReqOpts) (*protocol.ExtensionManifest, error) {
	url := url.URL{Path: "/developer/extensions"}

	bodyBytes, err := json.Marshal(manifest)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_extension_v2",
		StatSampleRate: defaultStatSampleRate,
	})

	var outManifest protocol.ExtensionManifest

	resp, err := c.DoJSON(ctx, &outManifest, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &outManifest, nil
}

func (c *clientImpl) TransitionExtensionState(ctx context.Context, extensionID, version, state string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/extensions/%s/%s", extensionID, version)}

	bodyBytes, err := json.Marshal(documents.StateTransition{
		State: state,
	})
	if err != nil {
		return err
	}

	req, err := c.NewRequest("PUT", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.transition_extension_state",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) ActivateExtensions(ctx context.Context, channelID string, config documents.ActivationConfigurationParams, reqOpts *twitchclient.ReqOpts) (*documents.ActivationsDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/activations", channelID)}

	bodyBytes, err := json.Marshal(config)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("PUT", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.extension_activations",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, c.handleErrorResponse(resp)
	}

	var document documents.ActivationsDocument
	err = json.NewDecoder(resp.Body).Decode(&document)
	if err != nil {
		return nil, fmt.Errorf("Unable to read response body: %s", err)
	}

	return &document, err
}

func (c *clientImpl) SetInstallationConfiguration(ctx context.Context, extensionID, version, channelID string, requiredConfigurationString string, reqOpts *twitchclient.ReqOpts) error {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/%s/%s/required_configuration", channelID, extensionID, version)}

	bodyBytes, err := json.Marshal(documents.RequiredInstallationConfiguration{
		RequiredConfigurationString: requiredConfigurationString,
	})
	if err != nil {
		return err
	}

	req, err := c.NewRequest("PUT", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.installation_configuration",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *clientImpl) SetFeatureFlags(ctx context.Context, extensionID, version, channelID string, flags documents.SetFeatureFlagsDocument, reqOpts *twitchclient.ReqOpts) (*documents.InstalledExtensionDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/channels/%s/extensions/%s/%s/features", channelID, extensionID, version)}

	bodyBytes, err := json.Marshal(flags)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("PUT", url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.set_feature_flags",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, c.handleErrorResponse(resp)
	}

	var document documents.InstalledExtensionDocument
	err = json.NewDecoder(resp.Body).Decode(&document)
	if err != nil {
		return nil, fmt.Errorf("Unable to read response body: %s", err)
	}

	return &document, err
}
