package client

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	gdserrors "code.justin.tv/gds/gds/golibs/errors"
)

func (c *clientImpl) handleErrorResponse(resp *http.Response) error {
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	clientErr, err := gdserrors.UnmarshalErrorForHTTP(body)
	if err != nil {
		return err
	}

	if code := gdserrors.GetHTTPStatusOrDefault(clientErr, 500); code >= 500 {
		return gdserrors.NewBuilder(resp.Status + ": server error").WithHTTPStatus(code).Build()
	}

	return clientErr
}

func (c *clientImpl) execute(ctx context.Context, req *http.Request, statName string, out interface{}, reqOpts *twitchclient.ReqOpts) error {
	opts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions." + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	defer close(req.Body)

	resp, err := c.Do(ctx, req, opts)
	if err != nil {
		return err
	}

	defer close(resp.Body)

	if err, found := gdserrors.ExtractFromHTTPResponse(resp); found {
		return err
	}

	if resp.StatusCode == http.StatusNoContent {
		out = nil
		return nil
	}

	if err = json.NewDecoder(resp.Body).Decode(out); err != nil {
		return fmt.Errorf("Unable to read response body: %s", err)
	}
	return nil
}

func close(c io.Closer) {
	if c != nil {
		_ = c.Close()
	}
}
