package metrics

import (
	"context"
	"net/http"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/instrumentation"
	goji "goji.io"
)

var srKey = new(int)

type fultonWrapper struct {
}

var _ instrumentation.ContextInstrumenter = (*fultonWrapper)(nil)

func newFultonInstrumenter() *fultonWrapper {
	return &fultonWrapper{}
}

// availabilityFromStatus returns the availability for a status.
func availabilityFromStatus(status int) telemetry.AvailabilityCode {
	if status >= 200 && status < 300 {
		return telemetry.AvailabilityCodeSucccess
	}
	if status < 500 {
		return telemetry.AvailabilityCodeClientError
	}
	return telemetry.AvailabilityCodeServerError
}

func (wrapper *fultonWrapper) Instrument(ctx context.Context, name string, status int, callTime time.Duration, rate float32) {
	sr := loadSampleReporter(ctx)

	sr.SampleBuilder.OperationName = name
	sr.SampleBuilder.Timestamp = time.Now()
	sr.OperationName = name
	sr.Timestamp = time.Now()

	sr.ReportDurationSample(telemetry.MetricDuration, callTime)
	sr.ReportAvailabilitySamples(availabilityFromStatus(status))
}

// CreateSampleReporterInstrumenter sets up middleware to report all our metrics using
// Fulton's SampleReporter
func CreateSampleReporterInstrumenter(server *goji.Mux, sr *telemetry.SampleReporter) {
	server.Use(storeSampleReporter(sr))

	rates := make(map[instrumentation.Frequency]float32)

	instrumentor := newFultonInstrumenter()

	dynamic := func(ctx context.Context) []instrumentation.ContextInstrumenter {
		return []instrumentation.ContextInstrumenter{instrumentor}
	}

	i := instrumentation.NewInstrumentor(rates, dynamic)
	server.Use(i.Middleware("UnknownFunction", instrumentation.Rare))
}

func storeSampleReporter(sr *telemetry.SampleReporter) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			timestamp := time.Now()
			ctx := context.WithValue(r.Context(), srKey, sr)
			ctx = telemetry.ContextWithTimestamp(ctx, timestamp)

			inner.ServeHTTP(w, r.WithContext(ctx))
		})
	}
}

func loadSampleReporter(ctx context.Context) *telemetry.SampleReporter {
	if sr, ok := ctx.Value(srKey).(*telemetry.SampleReporter); ok {
		return sr
	}
	return nil
}
