package shared

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/gds/gds/extensions"
)

const (
	// VideoOverlaySlotName is the singleton slot name assigned to video overlays, since there can be only one
	VideoOverlaySlotName string = "extension-overlay-1"
)

type AnchorType string
type AnchorSlot string

type AnchorParam interface {
	Type() AnchorType
	Slot() AnchorSlot
}

// TODO: Remove this when we can do proper wire protocol versioning.
// NullAnchorActivationParam is used to maintain backwards compatibility with the previous activationConfig struct
// where an inactive anchor would be returned as an empty config, rather than a null field.
type NullAnchorActivationParam struct {
	SlotName string `json:"slot"`
}

func (*NullAnchorActivationParam) Type() AnchorType   { return AnchorType("") }
func (p *NullAnchorActivationParam) Slot() AnchorSlot { return AnchorSlot("") }

type PanelAnchorActivationParam struct {
	SlotName string `json:"slot"`
}

func (*PanelAnchorActivationParam) Type() AnchorType   { return AnchorType("panel") }
func (p *PanelAnchorActivationParam) Slot() AnchorSlot { return AnchorSlot(p.SlotName) }

type VideoOverlayAnchorActivationParam struct {
	SlotName string `json:"slot"`
}

func (*VideoOverlayAnchorActivationParam) Type() AnchorType { return AnchorType("video_overlay") }
func (p *VideoOverlayAnchorActivationParam) Slot() AnchorSlot {
	return AnchorSlot(VideoOverlaySlotName)
}

type ComponentAnchorActivationParam struct {
	SlotName string `json:"slot"`
	X        int    `json:"x"`
	Y        int    `json:"y"`
}

func (*ComponentAnchorActivationParam) Type() AnchorType   { return AnchorType("component") }
func (p *ComponentAnchorActivationParam) Slot() AnchorSlot { return AnchorSlot(p.SlotName) }

type HiddenAnchorActivationParam struct {
	SlotName string `json:"slot"`
}

func (*HiddenAnchorActivationParam) Type() AnchorType   { return AnchorType("hidden") }
func (p *HiddenAnchorActivationParam) Slot() AnchorSlot { return AnchorSlot(p.SlotName) }

type AnyAnchorParam struct{ Value AnchorParam }

func (a AnyAnchorParam) String() string {
	return fmt.Sprintf("%+v", a.Value)
}

func (a AnyAnchorParam) MarshalJSON() ([]byte, error) {
	param := a.Value
	if param == nil {
		param = &NullAnchorActivationParam{}
	}

	data := make(map[string]interface{})
	bytes, err := json.Marshal(param)
	if err == nil {
		err = json.Unmarshal(bytes, &data)
	}
	if err != nil {
		return nil, err
	}
	data["anchor"] = string(param.Type())
	return json.Marshal(data)
}

func (a *AnyAnchorParam) UnmarshalJSON(bytes []byte) error {
	typer := struct {
		Type string `json:"anchor"`
	}{}
	if err := json.Unmarshal(bytes, &typer); err != nil {
		return err
	}

	a.Value = nil
	switch typer.Type {
	case "":
		// Handle the null anchor case separately from invalid anchors
		a.Value = &NullAnchorActivationParam{}
	case "component":
		a.Value = &ComponentAnchorActivationParam{}
	case "video_overlay":
		a.Value = &VideoOverlayAnchorActivationParam{}
	case "panel":
		a.Value = &PanelAnchorActivationParam{}
	case "hidden":
		a.Value = &HiddenAnchorActivationParam{}
	default:
		return extensions.ErrInvalidAnchor
	}
	return json.Unmarshal(bytes, &a.Value)
}
