package protocol

import (
	"time"

	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/twirputils"
	"code.justin.tv/extensions/orchestration/service/protocol/response"
	"code.justin.tv/extensions/orchestration/service/protocol/shared"
	emsdocs "code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/ems/protocol"
	emsprotocol "code.justin.tv/gds/gds/extensions/ems/protocol"
)

// FromEMSExtensionDocument makes a copy of an EMS extension document into our own document format.
func FromEMSExtensionDocument(extension *emsdocs.ExtensionDocument) *response.ExtensionDocument {
	resp := response.ExtensionDocument{
		Anchor:                extension.Anchor,
		AuthorName:            extension.AuthorName,
		BaseURI:               extension.BaseURI,
		BitsEnabled:           extension.BitsEnabled,
		CanInstall:            extension.CanInstall,
		ConfigURL:             extension.ConfigURL,
		ConfigurationLocation: extension.ConfigurationLocation,
		Description:           extension.Description,
		EULATOSURL:            extension.EULATOSURL,
		HasChatSupport:        extension.HasChatSupport,
		IconURL:               extension.IconURL,
		IconURLs:              map[string]string{},
		ID:                    extension.ID,
		InstallationCount:     extension.InstallationCount,
		LiveConfigURL:         extension.LiveConfigURL,
		Name:                  extension.Name,
		PanelHeight:           extension.PanelHeight,
		PrivacyPolicyURL:      extension.PrivacyPolicyURL,
		RequestIdentityLink:   extension.RequestIdentityLink,
		RequiredConfiguration: extension.RequiredConfiguration,
		SKU:                   extension.SKU,
		State:                 extension.State,
		Summary:               extension.Summary,
		SupportEmail:          extension.SupportEmail,
		VendorCode:            extension.VendorCode,
		Version:               extension.Version,
		ViewerURL:             extension.ViewerURL,
		ViewerURLs: response.ViewerURLs{
			Mobile:       extension.ViewerURLs.Mobile,
			Panel:        extension.ViewerURLs.Panel,
			VideoOverlay: extension.ViewerURLs.VideoOverlay,
			Component:    extension.ViewerURLs.Component,
			Hidden:       extension.ViewerURLs.Hidden,
		},
	}

	for _, url := range extension.AssetURLs {
		resp.AssetURLs = append(resp.AssetURLs, url)
	}

	for _, category := range extension.Categories {
		resp.Categories = append(resp.Categories, category)
	}

	for _, game := range extension.Games {
		resp.Games = append(resp.Games, game)
	}

	for k, v := range extension.IconURLs {
		resp.IconURLs[string(k)] = v
	}

	for _, ability := range extension.RequiredBroadcasterAbilities {
		resp.RequiredBroadcasterAbilities = append(resp.RequiredBroadcasterAbilities, ability)
	}

	for _, url := range extension.ScreenshotURLs {
		resp.ScreenshotURLs = append(resp.ScreenshotURLs, url)
	}

	for _, url := range extension.WhitelistedConfigURLs {
		resp.WhitelistedConfigURLs = append(resp.WhitelistedConfigURLs, url)
	}

	for _, url := range extension.WhitelistedPanelURLs {
		resp.WhitelistedPanelURLs = append(resp.WhitelistedPanelURLs, url)
	}

	if extension.Views.Mobile != nil {
		resp.Views.Mobile = &response.MobileExtensionView{
			CanLinkExternalContent: extension.Views.Mobile.CanLinkExternalContent,
			ViewerURL:              extension.Views.Mobile.ViewerURL,
		}
	}

	if extension.Views.Panel != nil {
		resp.Views.Panel = &response.PanelExtensionView{
			CanLinkExternalContent: extension.Views.Panel.CanLinkExternalContent,
			Height:                 extension.Views.Panel.Height,
			ViewerURL:              extension.Views.Panel.ViewerURL,
		}
	}

	if extension.Views.VideoOverlay != nil {
		resp.Views.VideoOverlay = &response.VideoOverlayExtensionView{
			CanLinkExternalContent: extension.Views.VideoOverlay.CanLinkExternalContent,
			ViewerURL:              extension.Views.VideoOverlay.ViewerURL,
		}
	}

	if extension.Views.Component != nil {
		resp.Views.Component = &response.ComponentExtensionView{
			AspectWidth:            extension.Views.Component.AspectWidth,
			AspectHeight:           extension.Views.Component.AspectHeight,
			CanLinkExternalContent: extension.Views.Component.CanLinkExternalContent,
			Size:                   extension.Views.Component.Size,
			ViewerURL:              extension.Views.Component.ViewerURL,
			Zoom:                   extension.Views.Component.Zoom,
			ZoomPixels:             extension.Views.Component.ZoomPixels,
		}
	}

	if extension.Views.Hidden != nil {
		resp.Views.Hidden = &response.HiddenExtensionView{
			CanLinkExternalContent: extension.Views.Hidden.CanLinkExternalContent,
			ViewerURL:              extension.Views.Hidden.ViewerURL,
		}
	}

	if extension.Views.Config != nil {
		resp.Views.Config = &response.ConfigExtensionView{
			CanLinkExternalContent: extension.Views.Config.CanLinkExternalContent,
			ViewerURL:              extension.Views.Config.ViewerURL,
		}
	}

	if extension.Views.LiveConfig != nil {
		resp.Views.LiveConfig = &response.LiveConfigExtensionView{
			CanLinkExternalContent: extension.Views.LiveConfig.CanLinkExternalContent,
			ViewerURL:              extension.Views.LiveConfig.ViewerURL,
		}
	}

	return &resp
}

func FromEMSInstalledExtensionDocument(installation *emsdocs.InstalledExtensionDocument) *response.InstalledExtensionDocument {
	installedExtension := response.InstalledExtensionDocument{
		Extension:          FromEMSExtensionDocument(installation.Extension),
		InstallationStatus: FromEMSInstallationStatusDocument(installation.InstallationStatus),
	}
	for _, action := range installation.RequiredActions {
		installedExtension.RequiredActions = append(installedExtension.RequiredActions, response.ActivationRequirement(string(action)))
	}
	return &installedExtension
}

func FromEMSInstallationStatusDocument(status *emsdocs.InstallationStatusDocument) *response.InstallationStatusDocument {
	resp := response.InstallationStatusDocument{
		ActivationState: status.ActivationState,
		CanActivate:     status.CanActivate,
		ID:              status.ID,
	}

	if status.Abilities != nil {
		resp.Abilities = &response.InstallationAbilitiesDocument{
			IsBitsEnabled: status.Abilities.IsBitsEnabled,
			IsChatEnabled: status.Abilities.IsChatEnabled,
		}
	}
	if status.PermittedFeatures != nil {
		resp.PermittedFeatures = &response.FeatureFlagsDocument{
			CanSendChat: status.PermittedFeatures.CanSendChat,
			CanUseBits:  status.PermittedFeatures.CanUseBits,
		}
	}
	if status.Anchor != nil {
		anchor := FromEMSAnyAnchorParam(*status.Anchor)

		resp.Anchor = &anchor
	}

	return &resp
}

func FromEMSAnyAnchorParam(anchor emsdocs.AnyAnchorParam) shared.AnyAnchorParam {
	ret := shared.AnyAnchorParam{}
	switch anchor.Value.Type() {
	case emsdocs.AnchorType("hidden"):
		ret.Value = &shared.HiddenAnchorActivationParam{
			SlotName: anchor.Value.(*emsdocs.HiddenAnchorActivationParam).SlotName,
		}
	case emsdocs.AnchorType("panel"):
		ret.Value = &shared.PanelAnchorActivationParam{
			SlotName: anchor.Value.(*emsdocs.PanelAnchorActivationParam).SlotName,
		}
	case emsdocs.AnchorType("video_overlay"):
		ret.Value = &shared.VideoOverlayAnchorActivationParam{
			SlotName: anchor.Value.(*emsdocs.VideoOverlayAnchorActivationParam).SlotName,
		}
	case emsdocs.AnchorType("component"):
		ret.Value = &shared.ComponentAnchorActivationParam{
			SlotName: anchor.Value.(*emsdocs.ComponentAnchorActivationParam).SlotName,
			X:        anchor.Value.(*emsdocs.ComponentAnchorActivationParam).X,
			Y:        anchor.Value.(*emsdocs.ComponentAnchorActivationParam).Y,
		}
	}
	return ret
}

func ToEMSAnyAnchorParam(anchor shared.AnyAnchorParam) emsdocs.AnyAnchorParam {
	ret := emsdocs.AnyAnchorParam{}
	switch anchor.Value.Type() {
	case shared.AnchorType("hidden"):
		ret.Value = &emsdocs.HiddenAnchorActivationParam{
			SlotName: anchor.Value.(*shared.HiddenAnchorActivationParam).SlotName,
		}
	case shared.AnchorType("panel"):
		ret.Value = &emsdocs.PanelAnchorActivationParam{
			SlotName: anchor.Value.(*shared.PanelAnchorActivationParam).SlotName,
		}
	case shared.AnchorType("video_overlay"):
		ret.Value = &emsdocs.VideoOverlayAnchorActivationParam{
			SlotName: anchor.Value.(*shared.VideoOverlayAnchorActivationParam).SlotName,
		}
	case shared.AnchorType("component"):
		ret.Value = &emsdocs.ComponentAnchorActivationParam{
			SlotName: anchor.Value.(*shared.ComponentAnchorActivationParam).SlotName,
			X:        anchor.Value.(*shared.ComponentAnchorActivationParam).X,
			Y:        anchor.Value.(*shared.ComponentAnchorActivationParam).Y,
		}
	}
	return ret
}

func ToEMSAssetInfo(assetInfo shared.AssetInfo) protocol.AssetInfo {
	return protocol.AssetInfo{
		ExtensionID: assetInfo.ExtensionID,
		AssetHash:   assetInfo.AssetHash,
		UploadedBy:  assetInfo.UploadedBy,
		UploadedAt:  assetInfo.UploadedAt,
		FileSize:    assetInfo.FileSize,
		FileName:    assetInfo.FileName,
		UploadID:    assetInfo.UploadID,
		BaseURI:     assetInfo.BaseURI,
	}
}

func FromDiscoveryCategory(cat *discovery.CategoryDocument) *response.CategoryDocument {
	return &response.CategoryDocument{
		Deleted:     cat.GetDeleted(),
		Description: cat.GetDescription(),
		ID:          cat.GetID(),
		Name:        cat.GetName(),
		Order:       cat.GetOrder(),
		Readonly:    cat.GetReadonly(),
		Slug:        twirputils.UnwrapStringValuePointer(cat.GetSlug()),
		SortKey:     cat.GetSortKey(),
		Type:        cat.GetType(),
		Visible:     cat.GetVisible(),
	}
}

func FromDiscoveryCategoryMembership(doc *discovery.ExtensionCategoryMembershipDocument) (*response.ExtensionCategoryMembershipDocument, error) {
	t, err := twirputils.UnwrapTimestampPointer(doc.GetStartTime())
	if err != nil {
		return nil, err
	}
	return &response.ExtensionCategoryMembershipDocument{
		CategoryID:  doc.GetCategoryID(),
		ExtensionID: doc.GetExtensionID(),
		Order:       twirputils.UnwrapFloatValuePointer(doc.GetOrder()),
		StartTime:   t,
	}, nil
}

func FromDiscoveryFeaturedCaarousels(doc *discovery.FeaturedCarouselsDocument) (*response.FeaturedCarouselsDocument, error) {
	carousels := make([]*response.FeaturedCarouselDocument, len(doc.GetCarousels()))
	for i, carousel := range doc.GetCarousels() {
		converted, err := FromDiscoveryFeaturedCarousel(carousel)
		if err != nil {
			return nil, err
		}
		carousels[i] = converted
	}

	return &response.FeaturedCarouselsDocument{
		Count:     int(doc.GetCount()),
		Carousels: carousels,
	}, nil
}

func FromDiscoveryFeaturedCarousel(doc *discovery.FeaturedCarouselDocument) (*response.FeaturedCarouselDocument, error) {
	t, err := twirputils.UnwrapTimestampPointer(doc.GetStartTime())
	if err != nil {
		return nil, err
	}
	return &response.FeaturedCarouselDocument{
		ScheduleID: doc.GetScheduleID(),
		StartTime:  *t,
		ID:         doc.GetID(),
		Entries:    FromDiscoveryFeaturedCarouselEntriesDocument(doc.GetEntries()),
	}, nil
}

func FromDiscoveryFeaturedCarouselEntriesDocument(doc *discovery.FeaturedCarouselEntriesDocument) *response.FeaturedCarouselEntriesDocument {
	entries := make([]*response.FeaturedCarouselEntryDocument, len(doc.GetCarouselEntries()))
	for i, entry := range doc.GetCarouselEntries() {
		entries[i] = FromDiscoveryFeaturedCarouselEntryDocument(entry)
	}
	return &response.FeaturedCarouselEntriesDocument{
		Entries: entries,
		Count:   int(doc.GetCount()),
	}
}

func FromDiscoveryFeaturedCarouselEntriesResponse(doc *discovery.GetFeaturedCarouselEntriesResponse) *response.FeaturedCarouselEntriesDocument {
	entries := make([]*response.FeaturedCarouselEntryDocument, len(doc.GetEntries()))
	for i, entry := range doc.GetEntries() {
		entries[i] = FromDiscoveryFeaturedCarouselEntryDocument(entry)
	}
	return &response.FeaturedCarouselEntriesDocument{
		Entries: entries,
		Count:   int(doc.GetCount()),
	}
}

func FromDiscoveryFeaturedCarouselEntryDocument(doc *discovery.FeaturedCarouselEntryDocument) *response.FeaturedCarouselEntryDocument {
	return &response.FeaturedCarouselEntryDocument{
		ID:         doc.GetID(),
		Order:      doc.GetOrder(),
		Content:    *FromDiscoveryFeaturedCarouselEntryContent(doc.GetContent()),
		CarouselID: doc.GetCarouselID(),
	}
}

func FromDiscoveryFeaturedCarouselEntryContent(doc *discovery.FeaturedCarouselEntryContent) *response.FeaturedCarouselEntryContent {
	return &response.FeaturedCarouselEntryContent{
		Title:        doc.GetTitle(),
		ClickThrough: doc.GetClickThrough(),
		ImageURL:     doc.GetImageURL(),
	}
}

func FromDiscoveryFeaturedSchedulesDocument(doc *discovery.FeaturedSchedulesDocument) (*response.FeaturedSchedulesDocument, error) {
	schedules := make([]*response.FeaturedScheduleDocument, len(doc.GetSchedules()))
	for i, schedule := range doc.GetSchedules() {
		scheduleDoc, err := FromDiscoveryFeaturedScheduleDocument(schedule)
		if err != nil {
			return nil, err
		}
		schedules[i] = scheduleDoc
	}

	return &response.FeaturedSchedulesDocument{
		Count:     int(doc.GetCount()),
		Schedules: schedules,
	}, nil
}

func FromDiscoveryFeaturedScheduleDocument(doc *discovery.FeaturedScheduleDocument) (*response.FeaturedScheduleDocument, error) {
	carousel, err := FromDiscoveryFeaturedCarousel(doc.GetCurrentCarousel())
	if err != nil {
		return nil, err
	}
	return &response.FeaturedScheduleDocument{
		Description:     doc.GetDescription(),
		Slug:            twirputils.UnwrapStringValuePointer(doc.GetSlug()),
		ID:              doc.GetID(),
		CurrentCarousel: carousel,
	}, nil
}

func FromDiscoveryExtensionGameMembership(doc *discovery.ExtensionGameMembershipDocument) *response.ExtensionGameMembershipDocument {
	return &response.ExtensionGameMembershipDocument{
		GameID:      int(doc.GetGameID()),
		ExtensionID: doc.GetExtensionID(),
	}
}

func FromDiscoveryCategories(doc *discovery.CategoriesDocument) *response.CategoriesDocument {
	categories := make([]*response.CategoryDocument, len(doc.GetCategories()))
	for i, category := range doc.GetCategories() {
		categories[i] = FromDiscoveryCategory(category)
	}

	return &response.CategoriesDocument{
		Count:      int(doc.GetCount()),
		Categories: categories,
	}
}

func FromDiscoveryCategoryExtensionsDocument(doc *discovery.CategoryExtensionsDocument) *response.CategoryExtensionsDocument {
	return &response.CategoryExtensionsDocument{
		TotalCount: doc.GetTotalCount(),
		IDs:        doc.GetIDs(),
	}
}

func FromDiscoveryCategoryOrderDocument(doc *discovery.CategoryOrderDocument) *response.CategoryOrderDocument {
	orders := make([]*response.CategoryOrder, len(doc.GetCategoryOrder()))
	for i, order := range doc.GetCategoryOrder() {
		orders[i] = &response.CategoryOrder{
			ID:    order.GetID(),
			Order: order.GetOrder(),
		}
	}

	return &response.CategoryOrderDocument{
		CategoryOrdering: orders,
	}
}

func ToEMSExtensionManifest(body *shared.ExtensionManifest) *emsprotocol.ExtensionManifest {
	manifest := emsprotocol.ExtensionManifest{
		ID:      body.ID,
		Version: body.Version,
		Format:  body.Format,

		AssetData: emsprotocol.AssetInfo{
			ExtensionID: body.AssetData.ExtensionID,
			AssetHash:   body.AssetData.AssetHash,
			UploadedBy:  body.AssetData.UploadedBy,
			FileSize:    body.AssetData.FileSize,
			FileName:    body.AssetData.FileName,
			UploadID:    body.AssetData.UploadID,
		},
		ExtensionRuntimeData: emsprotocol.ExtensionRuntime{
			BitsEnabled:                  body.ExtensionRuntimeData.BitsEnabled,
			BitsSupportLevel:             emsprotocol.BitsSupportLevel(body.ExtensionRuntimeData.BitsSupportLevel),
			ConfigurationLocation:        emsprotocol.ConfigurationLocation(body.ExtensionRuntimeData.ConfigurationLocation),
			HasChatSupport:               body.ExtensionRuntimeData.HasChatSupport,
			RequestIdentityLink:          body.ExtensionRuntimeData.RequestIdentityLink,
			RequiredBroadcasterAbilities: []string{},
			RequiredConfigurationString:  body.ExtensionRuntimeData.RequiredConfigurationString,
			SKU:                          body.ExtensionRuntimeData.SKU,
			SubscriptionsSupportLevel:    emsprotocol.SubscriptionsSupportLevel(body.ExtensionRuntimeData.SubscriptionsSupportLevel),
			VendorCode:                   body.ExtensionRuntimeData.VendorCode,
			WhitelistedConfigURLs:        []string{},
			WhitelistedPanelURLs:         []string{},
		},
		DiscoveryData: emsprotocol.DiscoveryMetadata{
			Summary:          body.DiscoveryData.Summary,
			Description:      body.DiscoveryData.Description,
			ScreenshotPaths:  []string{},
			ScreenshotURLs:   []string{},
			Categories:       []string{},
			EULATOSURL:       body.DiscoveryData.EULATOSURL,
			Games:            []int{},
			IconPaths:        make(map[emsprotocol.IconSize]string),
			IconURLs:         make(map[emsprotocol.IconSize]string),
			PrivacyPolicyURL: body.DiscoveryData.PrivacyPolicyURL,
			SupportEmail:     body.DiscoveryData.SupportEmail,
			Name:             body.DiscoveryData.Name,
			AuthorName:       body.DiscoveryData.AuthorName,
		},
		AccessControlData: emsprotocol.AccessControl{
			BroadcasterWhitelist: []string{},
			TestingAccounts:      []string{},
		},
		DeveloperData: emsprotocol.DeveloperInfo{
			AuthorEmail:    body.DeveloperData.AuthorEmail,
			TestingBaseURI: body.DeveloperData.TestingBaseURI,
			VersionState:   body.DeveloperData.VersionState,
		},
	}

	if body.AssetData.UploadedAt != nil {
		manifest.AssetData.UploadedAt = new(time.Time)
		*manifest.AssetData.UploadedAt = *body.AssetData.UploadedAt
	}

	if body.AssetData.BaseURI != nil {
		manifest.AssetData.BaseURI = new(string)
		*manifest.AssetData.BaseURI = *body.AssetData.BaseURI
	}

	for _, ability := range body.ExtensionRuntimeData.RequiredBroadcasterAbilities {
		manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities = append(manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities, ability)
	}

	for _, url := range body.ExtensionRuntimeData.WhitelistedConfigURLs {
		manifest.ExtensionRuntimeData.WhitelistedConfigURLs = append(manifest.ExtensionRuntimeData.WhitelistedConfigURLs, url)
	}

	for _, url := range body.ExtensionRuntimeData.WhitelistedPanelURLs {
		manifest.ExtensionRuntimeData.WhitelistedPanelURLs = append(manifest.ExtensionRuntimeData.WhitelistedPanelURLs, url)
	}

	if body.ExtensionRuntimeData.Views != nil {
		manifest.ExtensionRuntimeData.Views = &emsprotocol.ManifestViews{}
		if body.ExtensionRuntimeData.Views.Component != nil {
			manifest.ExtensionRuntimeData.Views.Component = &emsprotocol.ComponentManifest{
				AspectHeight: body.ExtensionRuntimeData.Views.Component.AspectHeight,
				AspectWidth:  body.ExtensionRuntimeData.Views.Component.AspectWidth,
				Size:         body.ExtensionRuntimeData.Views.Component.Size,
				ViewerPath:   body.ExtensionRuntimeData.Views.Component.ViewerPath,
				Zoom:         body.ExtensionRuntimeData.Views.Component.Zoom,
				ZoomPixels:   body.ExtensionRuntimeData.Views.Component.ZoomPixels,
			}
		}
		if body.ExtensionRuntimeData.Views.Config != nil {
			manifest.ExtensionRuntimeData.Views.Config = &emsprotocol.ConfigManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Config.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Hidden != nil {
			manifest.ExtensionRuntimeData.Views.Hidden = &emsprotocol.HiddenManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Hidden.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.LiveConfig != nil {
			manifest.ExtensionRuntimeData.Views.LiveConfig = &emsprotocol.LiveConfigManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.LiveConfig.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Mobile != nil {
			manifest.ExtensionRuntimeData.Views.Mobile = &emsprotocol.MobileManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Mobile.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Panel != nil {
			manifest.ExtensionRuntimeData.Views.Panel = &emsprotocol.PanelManifest{
				Height:     body.ExtensionRuntimeData.Views.Panel.Height,
				ViewerPath: body.ExtensionRuntimeData.Views.Panel.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.VideoOverlay != nil {
			manifest.ExtensionRuntimeData.Views.VideoOverlay = &emsprotocol.VideoOverlayManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath,
			}
		}
	}

	for _, path := range body.DiscoveryData.ScreenshotPaths {
		manifest.DiscoveryData.ScreenshotPaths = append(manifest.DiscoveryData.ScreenshotPaths, path)
	}

	for _, url := range body.DiscoveryData.ScreenshotURLs {
		manifest.DiscoveryData.ScreenshotURLs = append(manifest.DiscoveryData.ScreenshotURLs, url)
	}

	for _, category := range body.DiscoveryData.Categories {
		manifest.DiscoveryData.Categories = append(manifest.DiscoveryData.Categories, category)
	}

	for _, game := range body.DiscoveryData.Games {
		manifest.DiscoveryData.Games = append(manifest.DiscoveryData.Games, game)
	}

	for _, userid := range body.AccessControlData.BroadcasterWhitelist {
		manifest.AccessControlData.BroadcasterWhitelist = append(manifest.AccessControlData.BroadcasterWhitelist, userid)
	}

	for _, userid := range body.AccessControlData.TestingAccounts {
		manifest.AccessControlData.TestingAccounts = append(manifest.AccessControlData.TestingAccounts, userid)
	}

	if body.DeveloperData.LastTransitionedAt != nil {
		manifest.DeveloperData.LastTransitionedAt = new(time.Time)
		*manifest.DeveloperData.LastTransitionedAt = *body.DeveloperData.LastTransitionedAt
	}

	if body.DeveloperData.CreatedAt != nil {
		manifest.DeveloperData.CreatedAt = new(time.Time)
		*manifest.DeveloperData.CreatedAt = *body.DeveloperData.CreatedAt
	}

	for k, v := range body.DiscoveryData.IconPaths {
		manifest.DiscoveryData.IconPaths[emsprotocol.IconSize(k)] = v
	}

	for k, v := range body.DiscoveryData.IconURLs {
		manifest.DiscoveryData.IconURLs[emsprotocol.IconSize(k)] = v
	}

	return &manifest
}

func FromEMSExtensionManifest(body *emsprotocol.ExtensionManifest) *shared.ExtensionManifest {
	manifest := shared.ExtensionManifest{
		ID:      body.ID,
		Version: body.Version,
		Format:  body.Format,

		AssetData: shared.AssetInfo{
			ExtensionID: body.AssetData.ExtensionID,
			AssetHash:   body.AssetData.AssetHash,
			UploadedBy:  body.AssetData.UploadedBy,
			FileSize:    body.AssetData.FileSize,
			FileName:    body.AssetData.FileName,
			UploadID:    body.AssetData.UploadID,
		},
		ExtensionRuntimeData: shared.ExtensionRuntime{
			BitsEnabled:                  body.ExtensionRuntimeData.BitsEnabled,
			BitsSupportLevel:             string(body.ExtensionRuntimeData.BitsSupportLevel),
			ConfigurationLocation:        string(body.ExtensionRuntimeData.ConfigurationLocation),
			HasChatSupport:               body.ExtensionRuntimeData.HasChatSupport,
			RequestIdentityLink:          body.ExtensionRuntimeData.RequestIdentityLink,
			RequiredBroadcasterAbilities: []string{},
			RequiredConfigurationString:  body.ExtensionRuntimeData.RequiredConfigurationString,
			SKU:                          body.ExtensionRuntimeData.SKU,
			SubscriptionsSupportLevel:    string(body.ExtensionRuntimeData.SubscriptionsSupportLevel),
			VendorCode:                   body.ExtensionRuntimeData.VendorCode,
			WhitelistedConfigURLs:        []string{},
			WhitelistedPanelURLs:         []string{},
		},
		DiscoveryData: shared.DiscoveryMetadata{
			Summary:          body.DiscoveryData.Summary,
			Description:      body.DiscoveryData.Description,
			ScreenshotPaths:  []string{},
			ScreenshotURLs:   []string{},
			Categories:       []string{},
			EULATOSURL:       body.DiscoveryData.EULATOSURL,
			Games:            []int{},
			IconPaths:        make(map[string]string),
			IconURLs:         make(map[string]string),
			PrivacyPolicyURL: body.DiscoveryData.PrivacyPolicyURL,
			SupportEmail:     body.DiscoveryData.SupportEmail,
			Name:             body.DiscoveryData.Name,
			AuthorName:       body.DiscoveryData.AuthorName,
		},
		AccessControlData: shared.AccessControl{
			BroadcasterWhitelist: []string{},
			TestingAccounts:      []string{},
		},
		DeveloperData: shared.DeveloperInfo{
			AuthorEmail:    body.DeveloperData.AuthorEmail,
			TestingBaseURI: body.DeveloperData.TestingBaseURI,
			VersionState:   body.DeveloperData.VersionState,
		},
	}

	if body.AssetData.UploadedAt != nil {
		manifest.AssetData.UploadedAt = new(time.Time)
		*manifest.AssetData.UploadedAt = *body.AssetData.UploadedAt
	}

	if body.AssetData.BaseURI != nil {
		manifest.AssetData.BaseURI = new(string)
		*manifest.AssetData.BaseURI = *body.AssetData.BaseURI
	}

	for _, ability := range body.ExtensionRuntimeData.RequiredBroadcasterAbilities {
		manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities = append(manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities, ability)
	}

	for _, url := range body.ExtensionRuntimeData.WhitelistedConfigURLs {
		manifest.ExtensionRuntimeData.WhitelistedConfigURLs = append(manifest.ExtensionRuntimeData.WhitelistedConfigURLs, url)
	}

	for _, url := range body.ExtensionRuntimeData.WhitelistedPanelURLs {
		manifest.ExtensionRuntimeData.WhitelistedPanelURLs = append(manifest.ExtensionRuntimeData.WhitelistedPanelURLs, url)
	}

	if body.ExtensionRuntimeData.Views != nil {
		manifest.ExtensionRuntimeData.Views = &shared.ManifestViews{}
		if body.ExtensionRuntimeData.Views.Component != nil {
			manifest.ExtensionRuntimeData.Views.Component = &shared.ComponentManifest{
				AspectHeight: body.ExtensionRuntimeData.Views.Component.AspectHeight,
				AspectWidth:  body.ExtensionRuntimeData.Views.Component.AspectWidth,
				Size:         body.ExtensionRuntimeData.Views.Component.Size,
				ViewerPath:   body.ExtensionRuntimeData.Views.Component.ViewerPath,
				Zoom:         body.ExtensionRuntimeData.Views.Component.Zoom,
				ZoomPixels:   body.ExtensionRuntimeData.Views.Component.ZoomPixels,
			}
		}
		if body.ExtensionRuntimeData.Views.Config != nil {
			manifest.ExtensionRuntimeData.Views.Config = &shared.ConfigManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Config.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Hidden != nil {
			manifest.ExtensionRuntimeData.Views.Hidden = &shared.HiddenManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Hidden.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.LiveConfig != nil {
			manifest.ExtensionRuntimeData.Views.LiveConfig = &shared.LiveConfigManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.LiveConfig.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Mobile != nil {
			manifest.ExtensionRuntimeData.Views.Mobile = &shared.MobileManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.Mobile.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.Panel != nil {
			manifest.ExtensionRuntimeData.Views.Panel = &shared.PanelManifest{
				Height:     body.ExtensionRuntimeData.Views.Panel.Height,
				ViewerPath: body.ExtensionRuntimeData.Views.Panel.ViewerPath,
			}
		}
		if body.ExtensionRuntimeData.Views.VideoOverlay != nil {
			manifest.ExtensionRuntimeData.Views.VideoOverlay = &shared.VideoOverlayManifest{
				ViewerPath: body.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath,
			}
		}
	}

	for _, path := range body.DiscoveryData.ScreenshotPaths {
		manifest.DiscoveryData.ScreenshotPaths = append(manifest.DiscoveryData.ScreenshotPaths, path)
	}

	for _, url := range body.DiscoveryData.ScreenshotURLs {
		manifest.DiscoveryData.ScreenshotURLs = append(manifest.DiscoveryData.ScreenshotURLs, url)
	}

	for _, category := range body.DiscoveryData.Categories {
		manifest.DiscoveryData.Categories = append(manifest.DiscoveryData.Categories, category)
	}

	for _, game := range body.DiscoveryData.Games {
		manifest.DiscoveryData.Games = append(manifest.DiscoveryData.Games, game)
	}

	for _, userid := range body.AccessControlData.BroadcasterWhitelist {
		manifest.AccessControlData.BroadcasterWhitelist = append(manifest.AccessControlData.BroadcasterWhitelist, userid)
	}

	for _, userid := range body.AccessControlData.TestingAccounts {
		manifest.AccessControlData.TestingAccounts = append(manifest.AccessControlData.TestingAccounts, userid)
	}

	if body.DeveloperData.LastTransitionedAt != nil {
		manifest.DeveloperData.LastTransitionedAt = new(time.Time)
		*manifest.DeveloperData.LastTransitionedAt = *body.DeveloperData.LastTransitionedAt
	}

	if body.DeveloperData.CreatedAt != nil {
		manifest.DeveloperData.CreatedAt = new(time.Time)
		*manifest.DeveloperData.CreatedAt = *body.DeveloperData.CreatedAt
	}

	for k, v := range body.DiscoveryData.IconPaths {
		manifest.DiscoveryData.IconPaths[string(k)] = v
	}

	for k, v := range body.DiscoveryData.IconURLs {
		manifest.DiscoveryData.IconURLs[string(k)] = v
	}

	return &manifest
}

func FromDiscoveryMetadataToExtensionVersionDiscoveryDocument(extensionID, version string, d *protocol.DiscoveryMetadata) *discovery.ExtensionVersionDiscoveryDocument {
	games := make([]int64, len(d.Games))
	for i, g := range d.Games {
		games[i] = int64(g)
	}
	return &discovery.ExtensionVersionDiscoveryDocument{
		ExtensionID: extensionID,
		Version:     version,
		DiscoveryData: &discovery.DiscoveryDataDocument{
			Name:             d.Name,
			AuthorName:       d.AuthorName,
			Categories:       d.Categories,
			Description:      d.Description,
			EULATOSURL:       d.EULATOSURL,
			Games:            games,
			IconPaths:        d.IconPaths,
			IconURLs:         d.IconURLs,
			PrivacyPolicyURL: d.PrivacyPolicyURL,
			ScreenshotPaths:  d.ScreenshotPaths,
			ScreenshotURLs:   d.ScreenshotURLs,
			Summary:          d.Summary,
			SupportEmail:     d.SupportEmail,
			ViewerSummary:    d.ViewerSummary,
			Deleted:          false,
		},
	}
}
