package protocol

import (
	"testing"
	"time"

	"code.justin.tv/extensions/orchestration/service/protocol/shared"
	emsdocs "code.justin.tv/gds/gds/extensions/ems/documents"
	emsprotocol "code.justin.tv/gds/gds/extensions/ems/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCopyExtensionDocument(t *testing.T) {
	extension := emsdocs.ExtensionDocument{
		Anchor:                       "anchor",
		AssetURLs:                    []string{"a", "b", "c"},
		AuthorName:                   "Bob",
		BitsEnabled:                  true,
		Categories:                   []string{"z", "x", "c"},
		CanInstall:                   true,
		ConfigURL:                    "configurl",
		ConfigurationLocation:        "configlocation",
		Description:                  "description",
		EULATOSURL:                   "eula",
		Games:                        []int{234, 235, 725426, 1345, 35234},
		HasChatSupport:               false,
		IconURL:                      "iconurl",
		ID:                           "id",
		InstallationCount:            234,
		LiveConfigURL:                "liveconfig",
		Name:                         "name",
		PanelHeight:                  555,
		PrivacyPolicyURL:             "private",
		RequestIdentityLink:          true,
		RequiredBroadcasterAbilities: []string{"cb", "aasuhgaiesf"},
		RequiredConfiguration:        "requiredconfig",
		ScreenshotURLs:               []string{"screen1", "screen2"},
		SKU:                          "sku",
		State:                        "state",
		Summary:                      "summary",
		SupportEmail:                 "support",
		VendorCode:                   "vendor",
		Version:                      "version",
		ViewerURL:                    "viewer",
		ViewerURLs: emsdocs.ViewerURLs{
			Component:    "componet",
			Hidden:       "hidden",
			Mobile:       "mobile",
			Panel:        "panel",
			VideoOverlay: "video",
		},
		Views: emsdocs.ExtensionViews{
			Component: &emsdocs.ComponentExtensionView{
				AspectHeight:           3,
				AspectWidth:            4,
				CanLinkExternalContent: true,
				Size:                   3.14,
				ViewerURL:              "component",
				Zoom:                   true,
				ZoomPixels:             300,
			},
			Config: &emsdocs.ConfigExtensionView{
				CanLinkExternalContent: true,
				ViewerURL:              "config",
			},
			Hidden: &emsdocs.HiddenExtensionView{
				CanLinkExternalContent: true,
				ViewerURL:              "hidden",
			},
			LiveConfig: &emsdocs.LiveConfigExtensionView{
				CanLinkExternalContent: true,
				ViewerURL:              "liveconfig",
			},
			Mobile: &emsdocs.MobileExtensionView{
				CanLinkExternalContent: true,
				ViewerURL:              "mobile",
			},
			Panel: &emsdocs.PanelExtensionView{
				CanLinkExternalContent: true,
				Height:                 888,
				ViewerURL:              "panel",
			},
			VideoOverlay: &emsdocs.VideoOverlayExtensionView{
				CanLinkExternalContent: true,
				ViewerURL:              "overlay",
			},
		},
		WhitelistedConfigURLs: []string{"asdf", "hguiwherfa"},
		WhitelistedPanelURLs:  []string{"xcvkbjns", "uihsuraefs", "asuehfaesf"},
	}

	doc := FromEMSExtensionDocument(&extension)

	assert.Equal(t, doc.Anchor, extension.Anchor)
	assert.Equal(t, doc.AuthorName, extension.AuthorName)
	assert.Equal(t, doc.BitsEnabled, extension.BitsEnabled)
	assert.Equal(t, doc.CanInstall, extension.CanInstall)
	assert.Equal(t, doc.ConfigURL, extension.ConfigURL)
	assert.Equal(t, doc.ConfigurationLocation, extension.ConfigurationLocation)
	assert.Equal(t, doc.Description, extension.Description)
	assert.Equal(t, doc.EULATOSURL, extension.EULATOSURL)
	assert.Equal(t, doc.HasChatSupport, extension.HasChatSupport)
	assert.Equal(t, doc.IconURL, extension.IconURL)
	assert.Equal(t, doc.ID, extension.ID)
	assert.Equal(t, doc.InstallationCount, extension.InstallationCount)
	assert.Equal(t, doc.LiveConfigURL, extension.LiveConfigURL)
	assert.Equal(t, doc.Name, extension.Name)
	assert.Equal(t, doc.PanelHeight, extension.PanelHeight)
	assert.Equal(t, doc.PrivacyPolicyURL, extension.PrivacyPolicyURL)
	assert.Equal(t, doc.RequestIdentityLink, extension.RequestIdentityLink)
	assert.Equal(t, doc.RequiredBroadcasterAbilities, extension.RequiredBroadcasterAbilities)
	assert.Equal(t, doc.RequiredConfiguration, extension.RequiredConfiguration)
	assert.Equal(t, doc.SKU, extension.SKU)
	assert.Equal(t, doc.State, extension.State)
	assert.Equal(t, doc.Summary, extension.Summary)
	assert.Equal(t, doc.SupportEmail, extension.SupportEmail)
	assert.Equal(t, doc.VendorCode, extension.VendorCode)
	assert.Equal(t, doc.Version, extension.Version)
	assert.Equal(t, doc.Views.Component.AspectHeight, extension.Views.Component.AspectHeight)
	assert.Equal(t, doc.Views.Component.AspectWidth, extension.Views.Component.AspectWidth)
	assert.Equal(t, doc.Views.Component.CanLinkExternalContent, extension.Views.Component.CanLinkExternalContent)
	assert.Equal(t, doc.Views.Component.Size, extension.Views.Component.Size)
	assert.Equal(t, doc.Views.Component.ViewerURL, extension.Views.Component.ViewerURL)
	assert.Equal(t, doc.Views.Component.Zoom, extension.Views.Component.Zoom)
	assert.Equal(t, doc.Views.Component.ZoomPixels, extension.Views.Component.ZoomPixels)
	assert.Equal(t, doc.Views.Config.CanLinkExternalContent, extension.Views.Config.CanLinkExternalContent)
	assert.Equal(t, doc.Views.Config.ViewerURL, extension.Views.Config.ViewerURL)
	assert.Equal(t, doc.Views.LiveConfig.CanLinkExternalContent, extension.Views.LiveConfig.CanLinkExternalContent)
	assert.Equal(t, doc.Views.LiveConfig.ViewerURL, extension.Views.LiveConfig.ViewerURL)
	assert.Equal(t, doc.Views.Hidden.CanLinkExternalContent, extension.Views.Hidden.CanLinkExternalContent)
	assert.Equal(t, doc.Views.Hidden.ViewerURL, extension.Views.Hidden.ViewerURL)
	assert.Equal(t, doc.Views.VideoOverlay.CanLinkExternalContent, extension.Views.VideoOverlay.CanLinkExternalContent)
	assert.Equal(t, doc.Views.VideoOverlay.ViewerURL, extension.Views.VideoOverlay.ViewerURL)
	assert.Equal(t, doc.Views.Panel.CanLinkExternalContent, extension.Views.Panel.CanLinkExternalContent)
	assert.Equal(t, doc.Views.Panel.Height, extension.Views.Panel.Height)
	assert.Equal(t, doc.Views.Panel.ViewerURL, extension.Views.Panel.ViewerURL)

	assert.Equal(t, doc.ViewerURL, extension.ViewerURL)
	assert.Equal(t, doc.ViewerURLs.Component, extension.ViewerURLs.Component)
	assert.Equal(t, doc.ViewerURLs.Hidden, extension.ViewerURLs.Hidden)
	assert.Equal(t, doc.ViewerURLs.Mobile, extension.ViewerURLs.Mobile)
	assert.Equal(t, doc.ViewerURLs.Panel, extension.ViewerURLs.Panel)
	assert.Equal(t, doc.ViewerURLs.VideoOverlay, extension.ViewerURLs.VideoOverlay)

	require.Equal(t, len(doc.AssetURLs), len(extension.AssetURLs))
	require.Equal(t, len(doc.Categories), len(extension.Categories))
	require.Equal(t, len(doc.Games), len(extension.Games))
	require.Equal(t, len(doc.IconURLs), len(extension.IconURLs))
	require.Equal(t, len(doc.RequiredBroadcasterAbilities), len(extension.RequiredBroadcasterAbilities))
	require.Equal(t, len(doc.ScreenshotURLs), len(extension.ScreenshotURLs))
	require.Equal(t, len(doc.WhitelistedConfigURLs), len(extension.WhitelistedConfigURLs))
	require.Equal(t, len(doc.WhitelistedPanelURLs), len(extension.WhitelistedPanelURLs))

	for i := range extension.AssetURLs {
		assert.Equal(t, doc.AssetURLs[i], extension.AssetURLs[i])
	}
	for i := range extension.Categories {
		assert.Equal(t, doc.Categories[i], extension.Categories[i])
	}
	for i := range extension.Games {
		assert.Equal(t, doc.Games[i], extension.Games[i])
	}
	for i := range extension.RequiredBroadcasterAbilities {
		assert.Equal(t, doc.RequiredBroadcasterAbilities[i], extension.RequiredBroadcasterAbilities[i])
	}
	for i := range extension.ScreenshotURLs {
		assert.Equal(t, doc.ScreenshotURLs[i], extension.ScreenshotURLs[i])
	}
	for i := range extension.WhitelistedConfigURLs {
		assert.Equal(t, doc.WhitelistedConfigURLs[i], extension.WhitelistedConfigURLs[i])
	}
	for i := range extension.WhitelistedPanelURLs {
		assert.Equal(t, doc.WhitelistedPanelURLs[i], extension.WhitelistedPanelURLs[i])
	}
	for k := range extension.IconURLs {
		assert.Equal(t, doc.IconURLs[string(k)], extension.IconURLs[k])
	}
}

func TestFromEMSExtensionManifest(t *testing.T) {
	baseuri := "baseuri"
	now := time.Now()

	body := emsprotocol.ExtensionManifest{
		ID:      "id",
		Version: "version",
		Format:  1,

		AssetData: emsprotocol.AssetInfo{
			AssetHash:   "assethash",
			BaseURI:     &baseuri,
			ExtensionID: "extensionid",
			FileName:    "filename",
			FileSize:    "filesize",
			UploadID:    "uploadid",
			UploadedAt:  &now,
			UploadedBy:  "uploadedby",
		},
		ExtensionRuntimeData: emsprotocol.ExtensionRuntime{
			BitsEnabled:                  true,
			BitsSupportLevel:             emsprotocol.BitsSupportLevelRequired,
			ConfigurationLocation:        emsprotocol.Custom,
			HasChatSupport:               true,
			RequestIdentityLink:          true,
			RequiredBroadcasterAbilities: []string{"abilitya", "abilityb"},
			RequiredConfigurationString:  "requiredconfigurationstring",
			SKU:                          "sku",
			SubscriptionsSupportLevel:    emsprotocol.SubscriptionsSupportLevelOptional,
			VendorCode:                   "vendorcode",
			Views: &emsprotocol.ManifestViews{
				Component: &emsprotocol.ComponentManifest{
					AspectHeight: 4,
					AspectWidth:  3,
					Size:         100,
					ViewerPath:   "component_viewerpath",
					Zoom:         true,
					ZoomPixels:   234,
				},
				Config: &emsprotocol.ConfigManifest{
					ViewerPath: "config_viewerpath",
				},
				Hidden: &emsprotocol.HiddenManifest{
					ViewerPath: "hidden_viewerpath",
				},
				LiveConfig: &emsprotocol.LiveConfigManifest{
					ViewerPath: "liveconfig_viewerpath",
				},
				Mobile: &emsprotocol.MobileManifest{
					ViewerPath: "mobile_viewerpath",
				},
				Panel: &emsprotocol.PanelManifest{
					Height:     21342,
					ViewerPath: "panel_viewerpath",
				},
				VideoOverlay: &emsprotocol.VideoOverlayManifest{
					ViewerPath: "videooverlay_viewerpath",
				},
			},
			WhitelistedConfigURLs: []string{"url1", "url2"},
			WhitelistedPanelURLs:  []string{"url3", "url4"},
		},
		DiscoveryData: emsprotocol.DiscoveryMetadata{
			AuthorName:       "authorname",
			Categories:       []string{"cat1", "cat2"},
			Description:      "description",
			EULATOSURL:       "eula",
			Games:            []int{1, 2, 3},
			IconPaths:        make(map[emsprotocol.IconSize]string),
			IconURLs:         make(map[emsprotocol.IconSize]string),
			Name:             "name",
			PrivacyPolicyURL: "privacy",
			ScreenshotPaths:  []string{"path1", "path2"},
			ScreenshotURLs:   []string{"screenshoturl1", "screenshoturl2"},
		},
		AccessControlData: emsprotocol.AccessControl{
			BroadcasterWhitelist: []string{"broadcaster1", "broadcaster2"},
			TestingAccounts:      []string{"tester1", "tester2"},
		},
		DeveloperData: emsprotocol.DeveloperInfo{
			AuthorEmail:        "authoremail",
			CreatedAt:          &now,
			LastTransitionedAt: &now,
			TestingBaseURI:     "testingbaseuri",
			VersionState:       "versionstate",
		},
	}

	manifest := FromEMSExtensionManifest(&body)

	assert.Equal(t, body.ID, manifest.ID)
	assert.Equal(t, body.Version, manifest.Version)
	assert.Equal(t, body.Format, manifest.Format)
	assert.Equal(t, body.AssetData.ExtensionID, manifest.AssetData.ExtensionID)
	assert.Equal(t, body.AssetData.AssetHash, manifest.AssetData.AssetHash)
	assert.Equal(t, body.AssetData.UploadedBy, manifest.AssetData.UploadedBy)
	assert.Equal(t, *body.AssetData.UploadedAt, *manifest.AssetData.UploadedAt)
	assert.Equal(t, body.AssetData.FileSize, manifest.AssetData.FileSize)
	assert.Equal(t, body.AssetData.FileName, manifest.AssetData.FileName)
	assert.Equal(t, body.AssetData.UploadID, manifest.AssetData.UploadID)
	assert.Equal(t, *body.AssetData.BaseURI, *manifest.AssetData.BaseURI)
	assert.Equal(t, body.ExtensionRuntimeData.BitsEnabled, manifest.ExtensionRuntimeData.BitsEnabled)
	assert.Equal(t, string(body.ExtensionRuntimeData.BitsSupportLevel), manifest.ExtensionRuntimeData.BitsSupportLevel)
	assert.Equal(t, string(body.ExtensionRuntimeData.ConfigurationLocation), manifest.ExtensionRuntimeData.ConfigurationLocation)
	assert.Equal(t, body.ExtensionRuntimeData.HasChatSupport, manifest.ExtensionRuntimeData.HasChatSupport)
	assert.Equal(t, body.ExtensionRuntimeData.RequestIdentityLink, manifest.ExtensionRuntimeData.RequestIdentityLink)
	assert.Equal(t, body.ExtensionRuntimeData.RequiredBroadcasterAbilities, manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities)
	assert.Equal(t, body.ExtensionRuntimeData.RequiredConfigurationString, manifest.ExtensionRuntimeData.RequiredConfigurationString)
	assert.Equal(t, body.ExtensionRuntimeData.SKU, manifest.ExtensionRuntimeData.SKU)
	assert.Equal(t, string(body.ExtensionRuntimeData.SubscriptionsSupportLevel), manifest.ExtensionRuntimeData.SubscriptionsSupportLevel)
	assert.Equal(t, body.ExtensionRuntimeData.VendorCode, manifest.ExtensionRuntimeData.VendorCode)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.AspectHeight, manifest.ExtensionRuntimeData.Views.Component.AspectHeight)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.AspectWidth, manifest.ExtensionRuntimeData.Views.Component.AspectWidth)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.Size, manifest.ExtensionRuntimeData.Views.Component.Size)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.ViewerPath, manifest.ExtensionRuntimeData.Views.Component.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.Zoom, manifest.ExtensionRuntimeData.Views.Component.Zoom)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.ZoomPixels, manifest.ExtensionRuntimeData.Views.Component.ZoomPixels)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Config.ViewerPath, manifest.ExtensionRuntimeData.Views.Config.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Hidden.ViewerPath, manifest.ExtensionRuntimeData.Views.Hidden.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.LiveConfig.ViewerPath, manifest.ExtensionRuntimeData.Views.LiveConfig.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Mobile.ViewerPath, manifest.ExtensionRuntimeData.Views.Mobile.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Panel.Height, manifest.ExtensionRuntimeData.Views.Panel.Height)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Panel.ViewerPath, manifest.ExtensionRuntimeData.Views.Panel.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath, manifest.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath)
	assert.ElementsMatch(t, body.ExtensionRuntimeData.WhitelistedConfigURLs, manifest.ExtensionRuntimeData.WhitelistedConfigURLs)
	assert.ElementsMatch(t, body.ExtensionRuntimeData.WhitelistedPanelURLs, manifest.ExtensionRuntimeData.WhitelistedPanelURLs)
	assert.Equal(t, body.DiscoveryData.Summary, manifest.DiscoveryData.Summary)
	assert.Equal(t, body.DiscoveryData.Description, manifest.DiscoveryData.Description)
	assert.ElementsMatch(t, body.DiscoveryData.ScreenshotPaths, manifest.DiscoveryData.ScreenshotPaths)
	assert.ElementsMatch(t, body.DiscoveryData.ScreenshotURLs, manifest.DiscoveryData.ScreenshotURLs)
	assert.ElementsMatch(t, body.DiscoveryData.Categories, manifest.DiscoveryData.Categories)
	assert.Equal(t, body.DiscoveryData.EULATOSURL, manifest.DiscoveryData.EULATOSURL)
	assert.ElementsMatch(t, body.DiscoveryData.Games, manifest.DiscoveryData.Games)
	assert.Equal(t, body.DiscoveryData.PrivacyPolicyURL, manifest.DiscoveryData.PrivacyPolicyURL)
	assert.Equal(t, body.DiscoveryData.SupportEmail, manifest.DiscoveryData.SupportEmail)
	assert.Equal(t, body.DiscoveryData.Name, manifest.DiscoveryData.Name)
	assert.Equal(t, body.DiscoveryData.AuthorName, manifest.DiscoveryData.AuthorName)
	assert.Equal(t, body.AccessControlData.BroadcasterWhitelist, manifest.AccessControlData.BroadcasterWhitelist)
	assert.Equal(t, body.AccessControlData.TestingAccounts, manifest.AccessControlData.TestingAccounts)
	assert.Equal(t, body.DeveloperData.AuthorEmail, manifest.DeveloperData.AuthorEmail)
	assert.Equal(t, *body.DeveloperData.LastTransitionedAt, *manifest.DeveloperData.LastTransitionedAt)
	assert.Equal(t, *body.DeveloperData.CreatedAt, *manifest.DeveloperData.CreatedAt)
	assert.Equal(t, body.DeveloperData.TestingBaseURI, manifest.DeveloperData.TestingBaseURI)
	assert.Equal(t, body.DeveloperData.VersionState, manifest.DeveloperData.VersionState)

	assert.Equal(t, len(body.DiscoveryData.IconPaths), len(manifest.DiscoveryData.IconPaths))
	assert.Equal(t, len(body.DiscoveryData.IconURLs), len(manifest.DiscoveryData.IconURLs))

	for k, v := range body.DiscoveryData.IconPaths {
		w, ok := manifest.DiscoveryData.IconPaths[string(k)]
		assert.True(t, ok)
		assert.Equal(t, v, w)
	}

	for k, v := range body.DiscoveryData.IconURLs {
		w, ok := manifest.DiscoveryData.IconURLs[string(k)]
		assert.True(t, ok)
		assert.Equal(t, v, w)

	}

}

func TestToEMSExtensionManifest(t *testing.T) {
	baseuri := "baseuri"
	now := time.Now()

	body := shared.ExtensionManifest{
		ID:      "id",
		Version: "version",
		Format:  1,

		AssetData: shared.AssetInfo{
			AssetHash:   "assethash",
			BaseURI:     &baseuri,
			ExtensionID: "extensionid",
			FileName:    "filename",
			FileSize:    "filesize",
			UploadID:    "uploadid",
			UploadedAt:  &now,
			UploadedBy:  "uploadedby",
		},
		ExtensionRuntimeData: shared.ExtensionRuntime{
			BitsEnabled:                  true,
			BitsSupportLevel:             string(emsprotocol.BitsSupportLevelRequired),
			ConfigurationLocation:        string(emsprotocol.Custom),
			HasChatSupport:               true,
			RequestIdentityLink:          true,
			RequiredBroadcasterAbilities: []string{"abilitya", "abilityb"},
			RequiredConfigurationString:  "requiredconfigurationstring",
			SKU:                          "sku",
			SubscriptionsSupportLevel:    string(emsprotocol.SubscriptionsSupportLevelOptional),
			VendorCode:                   "vendorcode",
			Views: &shared.ManifestViews{
				Component: &shared.ComponentManifest{
					AspectHeight: 4,
					AspectWidth:  3,
					Size:         100,
					ViewerPath:   "component_viewerpath",
					Zoom:         true,
					ZoomPixels:   234,
				},
				Config: &shared.ConfigManifest{
					ViewerPath: "config_viewerpath",
				},
				Hidden: &shared.HiddenManifest{
					ViewerPath: "hidden_viewerpath",
				},
				LiveConfig: &shared.LiveConfigManifest{
					ViewerPath: "liveconfig_viewerpath",
				},
				Mobile: &shared.MobileManifest{
					ViewerPath: "mobile_viewerpath",
				},
				Panel: &shared.PanelManifest{
					Height:     21342,
					ViewerPath: "panel_viewerpath",
				},
				VideoOverlay: &shared.VideoOverlayManifest{
					ViewerPath: "videooverlay_viewerpath",
				},
			},
			WhitelistedConfigURLs: []string{"url1", "url2"},
			WhitelistedPanelURLs:  []string{"url3", "url4"},
		},
		DiscoveryData: shared.DiscoveryMetadata{
			AuthorName:       "authorname",
			Categories:       []string{"cat1", "cat2"},
			Description:      "description",
			EULATOSURL:       "eula",
			Games:            []int{1, 2, 3},
			IconPaths:        make(map[string]string),
			IconURLs:         make(map[string]string),
			Name:             "name",
			PrivacyPolicyURL: "privacy",
			ScreenshotPaths:  []string{"path1", "path2"},
			ScreenshotURLs:   []string{"screenshoturl1", "screenshoturl2"},
		},
		AccessControlData: shared.AccessControl{
			BroadcasterWhitelist: []string{"broadcaster1", "broadcaster2"},
			TestingAccounts:      []string{"tester1", "tester2"},
		},
		DeveloperData: shared.DeveloperInfo{
			AuthorEmail:        "authoremail",
			CreatedAt:          &now,
			LastTransitionedAt: &now,
			TestingBaseURI:     "testingbaseuri",
			VersionState:       "versionstate",
		},
	}

	manifest := ToEMSExtensionManifest(&body)

	assert.Equal(t, body.ID, manifest.ID)
	assert.Equal(t, body.Version, manifest.Version)
	assert.Equal(t, body.Format, manifest.Format)
	assert.Equal(t, body.AssetData.ExtensionID, manifest.AssetData.ExtensionID)
	assert.Equal(t, body.AssetData.AssetHash, manifest.AssetData.AssetHash)
	assert.Equal(t, body.AssetData.UploadedBy, manifest.AssetData.UploadedBy)
	assert.Equal(t, *body.AssetData.UploadedAt, *manifest.AssetData.UploadedAt)
	assert.Equal(t, body.AssetData.FileSize, manifest.AssetData.FileSize)
	assert.Equal(t, body.AssetData.FileName, manifest.AssetData.FileName)
	assert.Equal(t, body.AssetData.UploadID, manifest.AssetData.UploadID)
	assert.Equal(t, *body.AssetData.BaseURI, *manifest.AssetData.BaseURI)
	assert.Equal(t, body.ExtensionRuntimeData.BitsEnabled, manifest.ExtensionRuntimeData.BitsEnabled)
	assert.Equal(t, string(body.ExtensionRuntimeData.BitsSupportLevel), string(manifest.ExtensionRuntimeData.BitsSupportLevel))
	assert.Equal(t, string(body.ExtensionRuntimeData.ConfigurationLocation), string(manifest.ExtensionRuntimeData.ConfigurationLocation))
	assert.Equal(t, body.ExtensionRuntimeData.HasChatSupport, manifest.ExtensionRuntimeData.HasChatSupport)
	assert.Equal(t, body.ExtensionRuntimeData.RequestIdentityLink, manifest.ExtensionRuntimeData.RequestIdentityLink)
	assert.Equal(t, body.ExtensionRuntimeData.RequiredBroadcasterAbilities, manifest.ExtensionRuntimeData.RequiredBroadcasterAbilities)
	assert.Equal(t, body.ExtensionRuntimeData.RequiredConfigurationString, manifest.ExtensionRuntimeData.RequiredConfigurationString)
	assert.Equal(t, body.ExtensionRuntimeData.SKU, manifest.ExtensionRuntimeData.SKU)
	assert.Equal(t, string(body.ExtensionRuntimeData.SubscriptionsSupportLevel), string(manifest.ExtensionRuntimeData.SubscriptionsSupportLevel))
	assert.Equal(t, body.ExtensionRuntimeData.VendorCode, manifest.ExtensionRuntimeData.VendorCode)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.AspectHeight, manifest.ExtensionRuntimeData.Views.Component.AspectHeight)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.AspectWidth, manifest.ExtensionRuntimeData.Views.Component.AspectWidth)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.Size, manifest.ExtensionRuntimeData.Views.Component.Size)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.ViewerPath, manifest.ExtensionRuntimeData.Views.Component.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.Zoom, manifest.ExtensionRuntimeData.Views.Component.Zoom)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Component.ZoomPixels, manifest.ExtensionRuntimeData.Views.Component.ZoomPixels)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Config.ViewerPath, manifest.ExtensionRuntimeData.Views.Config.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Hidden.ViewerPath, manifest.ExtensionRuntimeData.Views.Hidden.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.LiveConfig.ViewerPath, manifest.ExtensionRuntimeData.Views.LiveConfig.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Mobile.ViewerPath, manifest.ExtensionRuntimeData.Views.Mobile.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Panel.Height, manifest.ExtensionRuntimeData.Views.Panel.Height)
	assert.Equal(t, body.ExtensionRuntimeData.Views.Panel.ViewerPath, manifest.ExtensionRuntimeData.Views.Panel.ViewerPath)
	assert.Equal(t, body.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath, manifest.ExtensionRuntimeData.Views.VideoOverlay.ViewerPath)
	assert.ElementsMatch(t, body.ExtensionRuntimeData.WhitelistedConfigURLs, manifest.ExtensionRuntimeData.WhitelistedConfigURLs)
	assert.ElementsMatch(t, body.ExtensionRuntimeData.WhitelistedPanelURLs, manifest.ExtensionRuntimeData.WhitelistedPanelURLs)
	assert.Equal(t, body.DiscoveryData.Summary, manifest.DiscoveryData.Summary)
	assert.Equal(t, body.DiscoveryData.Description, manifest.DiscoveryData.Description)
	assert.ElementsMatch(t, body.DiscoveryData.ScreenshotPaths, manifest.DiscoveryData.ScreenshotPaths)
	assert.ElementsMatch(t, body.DiscoveryData.ScreenshotURLs, manifest.DiscoveryData.ScreenshotURLs)
	assert.ElementsMatch(t, body.DiscoveryData.Categories, manifest.DiscoveryData.Categories)
	assert.Equal(t, body.DiscoveryData.EULATOSURL, manifest.DiscoveryData.EULATOSURL)
	assert.ElementsMatch(t, body.DiscoveryData.Games, manifest.DiscoveryData.Games)
	assert.Equal(t, body.DiscoveryData.PrivacyPolicyURL, manifest.DiscoveryData.PrivacyPolicyURL)
	assert.Equal(t, body.DiscoveryData.SupportEmail, manifest.DiscoveryData.SupportEmail)
	assert.Equal(t, body.DiscoveryData.Name, manifest.DiscoveryData.Name)
	assert.Equal(t, body.DiscoveryData.AuthorName, manifest.DiscoveryData.AuthorName)
	assert.Equal(t, body.AccessControlData.BroadcasterWhitelist, manifest.AccessControlData.BroadcasterWhitelist)
	assert.Equal(t, body.AccessControlData.TestingAccounts, manifest.AccessControlData.TestingAccounts)
	assert.Equal(t, body.DeveloperData.AuthorEmail, manifest.DeveloperData.AuthorEmail)
	assert.Equal(t, *body.DeveloperData.LastTransitionedAt, *manifest.DeveloperData.LastTransitionedAt)
	assert.Equal(t, *body.DeveloperData.CreatedAt, *manifest.DeveloperData.CreatedAt)
	assert.Equal(t, body.DeveloperData.TestingBaseURI, manifest.DeveloperData.TestingBaseURI)
	assert.Equal(t, body.DeveloperData.VersionState, manifest.DeveloperData.VersionState)

	assert.Equal(t, len(body.DiscoveryData.IconPaths), len(manifest.DiscoveryData.IconPaths))
	assert.Equal(t, len(body.DiscoveryData.IconURLs), len(manifest.DiscoveryData.IconURLs))

	for k, v := range body.DiscoveryData.IconPaths {
		w, ok := manifest.DiscoveryData.IconPaths[emsprotocol.IconSize(k)]
		assert.True(t, ok)
		assert.Equal(t, v, w)
	}

	for k, v := range body.DiscoveryData.IconURLs {
		w, ok := manifest.DiscoveryData.IconURLs[emsprotocol.IconSize(k)]
		assert.True(t, ok)
		assert.Equal(t, v, w)

	}

}

func TestFromDiscoveryMetadataToExtensionVersionDiscoveryDocument(t *testing.T) {

	extID := "fakeExtID"
	version := "fakeVersion"
	metadata := emsprotocol.DiscoveryMetadata{
		Summary:         "fakeSummary",
		ViewerSummary:   "fakeViewerSummary",
		Description:     "fakeDescription",
		ScreenshotPaths: []string{"fakeScreenshotPath1"},
		ScreenshotURLs:  []string{"fakeScreenshotURL1"},
		Categories:      []string{"fakeCategory1"},
		EULATOSURL:      "fakeEULATOSURL",
		Games:           []int{1},
		IconPaths: map[emsprotocol.IconSize]string{
			emsprotocol.Square24Icon:        "fakeSquare24IconPath",
			emsprotocol.Square100Icon:       "fakeSquare100IconPath",
			emsprotocol.DiscoverySplashIcon: "fakeDiscoverySplashIcon",
		},
		IconURLs: map[emsprotocol.IconSize]string{
			emsprotocol.Square24Icon:        "fakeSquare24IconURL",
			emsprotocol.Square100Icon:       "fakeSquare100IconURL",
			emsprotocol.DiscoverySplashIcon: "fakeDiscoverySplashURL",
		},
		PrivacyPolicyURL: "fakePrivacyPolicyURL",
		SupportEmail:     "fakeSupportEmail",
		Name:             "fakeName",
		AuthorName:       "fakeAuthorName",
	}

	actual := FromDiscoveryMetadataToExtensionVersionDiscoveryDocument(extID, version, &metadata)

	assert.Equal(t, extID, actual.ExtensionID)
	assert.Equal(t, version, actual.Version)
	assert.Equal(t, metadata.Summary, actual.DiscoveryData.Summary)
	assert.Equal(t, metadata.ViewerSummary, actual.DiscoveryData.ViewerSummary)
	assert.Equal(t, metadata.Description, actual.DiscoveryData.Description)
	assert.Len(t, actual.DiscoveryData.ScreenshotPaths, len(metadata.ScreenshotPaths))
	assert.Equal(t, metadata.ScreenshotPaths[0], actual.DiscoveryData.ScreenshotPaths[0])
	assert.Len(t, actual.DiscoveryData.ScreenshotURLs, len(metadata.ScreenshotURLs))
	assert.Equal(t, metadata.ScreenshotURLs[0], actual.DiscoveryData.ScreenshotURLs[0])
	assert.Len(t, actual.DiscoveryData.Categories, len(metadata.Categories))
	assert.Equal(t, metadata.Categories[0], actual.DiscoveryData.Categories[0])
	assert.Equal(t, metadata.EULATOSURL, actual.DiscoveryData.EULATOSURL)
	assert.Len(t, actual.DiscoveryData.Games, len(metadata.Games))
	assert.Equal(t, int64(metadata.Games[0]), actual.DiscoveryData.Games[0])
	assert.Len(t, actual.DiscoveryData.IconPaths, len(metadata.IconPaths))
	for k, v := range metadata.IconPaths {
		assert.Equal(t, v, actual.DiscoveryData.IconPaths[string(k)])
	}
	assert.Len(t, actual.DiscoveryData.IconURLs, len(metadata.IconURLs))
	for k, v := range metadata.IconURLs {
		assert.Equal(t, v, actual.DiscoveryData.IconURLs[string(k)])
	}
	assert.Equal(t, metadata.PrivacyPolicyURL, actual.DiscoveryData.PrivacyPolicyURL)
	assert.Equal(t, metadata.SupportEmail, actual.DiscoveryData.SupportEmail)
	assert.Equal(t, metadata.Name, actual.DiscoveryData.Name)
	assert.Equal(t, metadata.AuthorName, actual.DiscoveryData.AuthorName)
}
