package routes

import (
	"fmt"
	"net/http"

	cartman "code.justin.tv/web/cartman/client"
	goji "goji.io"
	"goji.io/pat"
)

// Inject initializes the route middleware and adds the handler to the mux.
func (r *Route) Inject(mux *goji.Mux) {
	h := r.authMiddleware(r.Handler)
	patMethod := httpMethodPattern(r.Method)

	// handle some route ending variations
	for _, suffix := range []string{".json/", ".json", "/", ""} {
		mux.Handle(patMethod(r.Pattern+suffix), h)
	}
}

func (r *Route) authMiddleware(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {

		ctx := req.Context()

		hasCapabilities := len(r.mandatoryCapabilities) > 0 || len(r.optionalCapabilities) > 0
		allowAnonymous := r.allowAnonymous || !hasCapabilities
		isAnonymous := !cartman.HasAuthenticationCredentials(ctx, req)

		if isAnonymous && allowAnonymous {
			h.ServeHTTP(w, req.WithContext(ctx))
			return
		}

		// TODO Handle routes that require login / capabilities
		// Currently no routes have capabilities, so all routes will appear "anonymous"
		// so we should never get here.

		w.WriteHeader(http.StatusUnauthorized)
	})
}

func httpMethodPattern(method string) func(pat string) *pat.Pattern {
	switch method {
	case http.MethodGet:
		return pat.Get
	case http.MethodPost:
		return pat.Post
	case http.MethodPut:
		return pat.Put
	case http.MethodDelete:
		return pat.Delete
	default:
		panic(fmt.Errorf("a route's method must be one of GET, PUT, POST, or DELETE. Got %q instead", method))
	}
}
