package routes

import "net/http"

// Route represents a single orchestrator route, allowing us to
// associate additional data with it (such as required capabilities)
type Route struct {
	Method  string
	Pattern string
	Handler http.Handler

	mandatoryCapabilities []string
	optionalCapabilities  []string
	allowAnonymous        bool
}

// MandatoryCapabilities configures this route with a list of
// named capabilities that must be granted by the auth system
// for the route to be dispatched.  If any of the requested
// capabilities are not granted, the route will immediately
// return 401
//
// Granted capabilities will be inclued in an HTTP header
// for the downstream services to verify their presence.
// This is not redundant; it prevents anyone with direct access
// to the downstream services from making calls to those services
// bypassing the orchestrator.
func (r *Route) MandatoryCapabilities(capabilities ...string) *Route {
	r.mandatoryCapabilities = capabilities
	return r
}

// OptionalCapabilities configures this route with a list of
// named capabilities that can optionally be granted by the
// auth system.  Any mandatory and optional capabilities
// will be combined in the HTTP header for the request
// to the downstream services
func (r *Route) OptionalCapabilities(capabilities ...string) *Route {
	r.optionalCapabilities = capabilities
	return r
}

// AllowAnonymous can be used after Capabilities to configure the auth
// middleware to also allow unauthenticated requests.
func (r *Route) AllowAnonymous() *Route {
	r.allowAnonymous = true
	return r
}
