package routes

import (
	"net/http"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	discotwirp "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	htmid "code.justin.tv/gds/gds/golibs/http"

	"code.justin.tv/extensions/instrumentation"
	"code.justin.tv/extensions/orchestration/service/api"
	"code.justin.tv/extensions/orchestration/service/clients"
	"code.justin.tv/extensions/orchestration/service/config"
	"code.justin.tv/extensions/orchestration/service/metrics"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	gdsems "code.justin.tv/gds/gds/extensions/ems/client"
	"code.justin.tv/infosec/cors"
	"goji.io/pat"

	goji "goji.io"
)

var corsPolicy = cors.Policy{
	AllowedOrigins: cors.Origins("*"),
	AllowMethods:   cors.Methods("GET", "POST"),
	AllowHeaders:   cors.Headers("Accept", "Authorization", "Content-Type"),
}

// Server holds all the routes for the orhestrator server
type Server struct {
	Mux    *goji.Mux
	routes []*Route
}

func (s *Server) Get(pattern, name string, handler htmid.APICall) *Route {
	// fulton stats don't use the frequency, so any value here will do
	wrappedHandler := htmid.NewAPIHandler(name, instrumentation.Frequent, handler)
	return s.registerRoute(http.MethodGet, pattern, wrappedHandler)
}

func (s *Server) Post(pattern, name string, handler htmid.APICall) *Route {
	// fulton stats don't use the frequency, so any value here will do
	wrappedHandler := htmid.NewAPIHandler(name, instrumentation.Frequent, handler)

	return s.registerRoute(http.MethodPost, pattern, wrappedHandler)
}

func (s *Server) Put(pattern, name string, handler htmid.APICall) *Route {
	// fulton stats don't use the frequency, so any value here will do
	wrappedHandler := htmid.NewAPIHandler(name, instrumentation.Frequent, handler)

	return s.registerRoute(http.MethodPut, pattern, wrappedHandler)
}

func (s *Server) Delete(pattern, name string, handler htmid.APICall) *Route {
	// fulton stats don't use the frequency, so any value here will do
	wrappedHandler := htmid.NewAPIHandler(name, instrumentation.Frequent, handler)

	return s.registerRoute(http.MethodDelete, pattern, wrappedHandler)
}

func (s *Server) registerRoute(method, pattern string, handler http.Handler) *Route {
	r := &Route{
		Handler: handler,
		Method:  method,
		Pattern: pattern,
	}
	s.routes = append(s.routes, r)
	return r
}

// BuildServer creates the server routes
func BuildServer(a *api.API, conf *config.Config, sr *telemetry.SampleReporter) *Server {
	server := &Server{
		Mux: twitchserver.NewServer(),
	}

	server.Mux.Use(corsPolicy.MustMiddleware)
	server.Mux.Use(cors.BlockOnOptions)
	server.Mux.Use(config.Middleware(conf))

	emsClient, err := gdsems.NewClient(twitchclient.ClientConf{
		Host: conf.EMSURL,
	})

	if err != nil {
		panic(err)
	}

	discoveryClient := discotwirp.NewDiscoveryProtobufClient(conf.DiscoURL, twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host: conf.DiscoURL,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 200,
		},
	}))

	if err != nil {
		panic(err)
	}

	server.Mux.Use(clients.WithClients(emsClient, discoveryClient))
	metrics.CreateSampleReporterInstrumenter(server.Mux, sr)

	server.CreateRoutes()

	for _, r := range server.routes {
		r.Inject(server.Mux)
	}

	// default fallbacks in case we miss something.  TODO: Alarm on these so we
	// know orchestrator is incomplete
	server.Mux.HandleFunc(pat.Delete("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Get("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Head("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Options("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Patch("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Post("/*"), api.Proxy)
	server.Mux.HandleFunc(pat.Put("/*"), api.Proxy)

	return server
}
