data "aws_acm_certificate" "orchestrator_ssl_certificate" {
  #
  # The domain needs to be a wildcard domain so users of your service can create their own DNS entries
  # pointing to the VPC Endpoints in their own VPCs that connect up to your LB, so SSL hostname validation
  # works.
  # e.g. they will want to create ${vpc_id}.staging.ext-discovery.us-west-2.internal.justin.tv IN CNAME
  #       ${their_vpc_endpoint_hostname}
  # so their services can resolve something that points to the interface in their own account that is connected
  # to your account.
  #
  domain = "*.${var.environment}.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  statuses    = ["ISSUED"]
  most_recent = true
}

module "orchestrator" {
  source = "git::ssh://git@git-aws.internal.justin.tv/gds/terraform-fargate_service_template"

  ssl_cert_arn           = data.aws_acm_certificate.orchestrator_ssl_certificate.arn
  build_security_group   = data.aws_security_group.default.id
  ingress_security_group = data.terraform_remote_state.vpc_setup.outputs.us_west_2_twitch_subnets_sg_id
  private_subnets        = data.terraform_remote_state.vpc_setup.outputs.us_west_2_private_subnets
  public_subnets         = data.terraform_remote_state.vpc_setup.outputs.us_west_2_public_subnets
  vpc_id                 = data.terraform_remote_state.vpc_setup.outputs.us_west_2_vpc_id
  service                = var.service
  owner                  = var.owner
  component              = var.component
  environment            = var.environment
  account_name           = var.account_name

  # How many containers to spin up behind the NLB
  ecs_min_container_count = "10"
  ecs_max_container_count = "30"
  enable_auto_scaling     = "true"
  healthcheck_port        = "8080"
  healthcheck_protocol    = "HTTP"
  container_port          = "8443"
  container_protocol      = "TLS"
  nlb_test_hostname       = "test.production.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  # Use EC2 instead of FARGATE
  launch_type            = "EC2"
  ecs_min_instance_count = "10"
  ecs_max_instance_count = "30"

  # Slack configuration for the lambda that will send messages to a slack channel to request approval
  # for deploys.

  slack_channel     = "#ext-anchors-oncall"
  slack_username    = "orchestratorDeployer"
  slack_webhook_url = var.slack_webhook_url

  # There are no prod users of the ext/orchestrator vpc service endpoint yet
  whitelisted_vpc_endpoint_users = [
    "arn:aws:iam::523543649671:root",
  ]
}

# Upload jenkins artifacts to this bucket
output "orchestrator_jenkins_artifact_bucket" {
  value = module.orchestrator.jenkins_artifact_bucket_name
}

# And with this filename
output "orchestrator_jenkins_artifact_name" {
  value = module.orchestrator.jenkins_artifact_filename
}

# This is the role arn that we will attach iam policies to so the server can use other resources
output "orchestrator_iam_role_name" {
  value = module.orchestrator.ecs_task_iam_role_name
}

output "orchestrator_endpoint_service_name" {
  value = module.orchestrator.service_endpoint_service_name
}

##
## -- connect the alerts sns topic to pagerduty
##
resource "aws_sns_topic_subscription" "alerts" {
  topic_arn              = module.orchestrator.alerts_sns_topic_arn
  endpoint_auto_confirms = true
  protocol               = "https"
  endpoint               = "https://events.pagerduty.com/integration/${var.pagerduty_integration_key}/enqueue"
}


data "aws_iam_policy_document" "s2s_assume_role" {
  statement {
    sid = "1"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::180116294062:role/malachai/*"
    ]
  }
}

resource "aws_iam_policy" "s2s_policy" {
  name = "orchestrator-ecs-task-s2s-policy"
  policy = "${data.aws_iam_policy_document.s2s_assume_role.json}"
}

resource "aws_iam_role_policy_attachment" "s2s-attach" {
  role       = "${module.orchestrator.ecs_task_iam_role_name}"
  policy_arn = "${aws_iam_policy.s2s_policy.arn}"
}
