package c7

import (
	"fmt"
	"strings"

	"reflect"

	"code.justin.tv/amzn/C7-go/internal/ast"
	"code.justin.tv/amzn/C7-go/internal/rule"
	"io"
)

// NewC7 creates a new C7 object that is scoped to rules
// matching the selectors passed in following preference order.
// Selectors: should be a list of strings matching the selectors in the
//   configs that this C7 is operating on.
func NewC7(set C7Set, selectors ...string) *C7 {
	return &C7{
		rules:     set.rules,
		selectors: selectors,
	}
}

// C7 is a scoped set of rules
type C7 struct {
	rules     []*rule.Rule
	selectors []string
}

// matchingRule returns the rule that best matches the namespace and keys
// The general match is the more specific rule wins, eg a.b beats *.* on scopes
// and in a tie, the match weights higher on the lhs of the selector
// and on later rules
func (c *C7) matchingRule(namespace string, keys ...string) *rule.Rule {
	var matches []*rule.Rule

	for _, r := range c.rules {
		if !r.Matches(namespace, keys...) {
			continue
		}

		matches = append(matches, r)
	}

	if len(matches) == 0 {
		return nil
	}

	var match *rule.Rule
	score := -1

RULES:
	for _, rule := range matches {
		thisScore := 0
		selectors := rule.Selectors
		if len(selectors) != len(c.selectors) { // mismatch == 0 score
			thisScore = 0
		} else {
			for i := 0; i < len(selectors); i++ {
				ruleSelector := selectors[i]
				mySelector := c.selectors[i]
				if ruleSelector == "*" {
					continue
				}
				if ruleSelector == mySelector {
					thisScore++
					continue
				}
				continue RULES
			}
		}
		if thisScore > score {
			match = rule
			score = thisScore
		}
	}
	return match
}

// HasValue returns true if there is a value in the config for the namespace and keys
func (c *C7) HasValue(namespace string, keys ...string) bool {
	match := c.matchingRule(namespace, keys...)
	return match != nil
}

func typeError(expectedType string, rule *rule.Rule) error {
	return fmt.Errorf(
		"failed to get %s:%s, want: %s, have: %s",
		rule.Namespace,
		strings.Join(rule.Keys, "."),
		expectedType,
		rule.Value.StrType())
}

// Int64 returns an int64 value if present
func (c *C7) int64(namespace string, keys ...string) (*int64, error) {
	match := c.matchingRule(namespace, keys...)
	if match == nil {
		return nil, nil
	}
	if match.Value.IntValue == nil {
		return nil, typeError("int64", match)
	}
	return match.Value.IntValue, nil
}

// Int64 returns an int64 value if present
func (c *C7) duration(namespace string, keys ...string) (int64, error) {
	match := c.matchingRule(namespace, keys...)
	if match == nil {
		return 0, nil
	}
	if match.Value.DurationValue == nil {
		return 0, typeError("time.Duration", match)
	}
	return int64(*match.Value.DurationValue), nil
}

// String returns a string if present
func (c *C7) string(namespace string, keys ...string) (*string, error) {
	match := c.matchingRule(namespace, keys...)
	if match == nil {
		return nil, nil
	}
	if match.Value.StrValue == nil {
		return nil, typeError("string", match)
	}
	return match.Value.StrValue, nil
}

// Bool returns a bool if present
func (c *C7) bool(namespace string, keys ...string) (*bool, error) {
	match := c.matchingRule(namespace, keys...)
	if match == nil {
		return nil, nil
	}
	if match.Value.BoolValue == nil {
		return nil, typeError("bool", match)
	}
	return match.Value.BoolValue, nil
}

func fieldError(sType reflect.Type, fType reflect.StructField, err error) error {
	return fmt.Errorf(
		`failed to fill %s.%s: %s`,
		sType.Name(),
		fType.Name,
		err)
}

// FillWithNamespace takes a properly annotated struct (see README)
// and a namespace prefix and will fill the struct in with the appropriate values
// or error on type mismatches
func (c *C7) FillWithNamespace(namespace string, i interface{}) error {
	typ := reflect.TypeOf(i)
	if typ.Kind() != reflect.Ptr {
		return fmt.Errorf("c7 can only fill pointers, passed %q", typ.Kind())
	}

	structType := typ.Elem()
	if structType.Kind() != reflect.Struct {
		return fmt.Errorf("c7 can only fill structs, passed %q", structType.Kind())
	}

	structVal := reflect.ValueOf(i).Elem()

	for i := 0; i < structType.NumField(); i++ {
		fieldType := structType.Field(i)
		fieldValue := structVal.Field(i)

		c7Tag := fieldType.Tag.Get("c7")

		if c7Tag == "" {
			return fmt.Errorf(
				`struct tag must provide a key, tag: "", field: %s.%s`, structType.Name(), fieldType.Name)
		}

		if strings.Contains(c7Tag, ":") {
			return fmt.Errorf(`namespaces are not yet supported in tags: %q`, c7Tag)
		}

		keys := strings.Split(c7Tag, ".")
		if len(keys) != 1 {
			return fmt.Errorf(`invalid keys: %q`, c7Tag)
		}

		if !c.HasValue(namespace, keys...) {
			continue
		}

		if !fieldValue.IsValid() {
			return fmt.Errorf("field %s.%s is invalid", structType, fieldType.Name)
		}

		if !fieldValue.CanSet() {
			return fmt.Errorf("field %s.%s can't be set", structType, fieldType.Name)
		}

		if fieldType.Type.Kind() == reflect.String {
			val, err := c.string(namespace, keys...)
			if err != nil {
				return fieldError(structType, fieldType, err)
			}
			fieldValue.SetString(*val)
			continue
		} else if fieldType.Type.Kind() == reflect.Bool {
			val, err := c.bool(namespace, keys...)
			if err != nil {
				return fieldError(structType, fieldType, err)
			}
			fieldValue.SetBool(*val)
			continue
		} else if fieldType.Type.Kind() == reflect.Int64 {
			if fieldType.Type.String() == "time.Duration" {
				val, err := c.duration(namespace, keys...)
				if err != nil {
					return fieldError(structType, fieldType, err)
				}
				fieldValue.SetInt(val)
				continue
			}
			val, err := c.int64(namespace, keys...)
			if err != nil {
				return fieldError(structType, fieldType, err)
			}
			fieldValue.SetInt(*val)
			continue
		} else {
			return fmt.Errorf(
				`attempted to set Config.Value for unsupported field type: %s`, fieldType.Type)
		}
	}

	return nil
}

// C7Set is a group of rules not reduced by selectors, eg it is the raw unprocessed rules
//  it is generally used as a holder for rules that will be further scoped down into a C7
//  or is used by an aggregagtor or resolver
type C7Set struct {
	rules rule.Rules
}

// NewC7Set creates a new empty set
func NewC7Set() *C7Set {
	return &C7Set{
		rules: make(rule.Rules, 0),
	}
}

// Parse loads C7Set from file
func Parse(data string) (*C7Set, error) {
	rules, err := ast.Parse(data)
	if err != nil {
		return nil, err
	}
	return &C7Set{rules: rules}, nil
}

// Write exports the list to a Writer, used for serialization
func (c *C7Set) Write(w io.Writer) error {
	for _, rule := range c.rules {
		fmt.Fprintf(w, "%s\n", rule.String())
	}
	return nil
}

func (c *C7Set) addRule(rule *rule.Rule) error {
	/* This is N^2 but I'm lazy. If this becomes too slow we can easily
	* add a set of maps, one per selector to check for existence and replacement to speed this up */
	for i, existingRule := range c.rules {
		if !rule.EqualsIgnoringValue(existingRule) {
			continue
		}
		t1 := existingRule.StrType()
		t2 := rule.StrType()
		if t1 != t2 {
			return fmt.Errorf("Trying to replace %s type %s with type %s", existingRule.PrefixString(), t1, t2)
		}

		c.rules[i] = rule
		return nil
	}
	c.rules = append(c.rules, rule)
	return nil
}

// Merge appends additional rules on top of these. The appended rules supercede the current rules
func (c *C7Set) Merge(set *C7Set) error {
	if set == nil {
		return nil
	}

	for _, rule := range set.rules {
		err := c.addRule(rule)
		if err != nil {
			return err
		}
	}
	return nil
}

// Size returns the size of the rules array
func (c *C7Set) Size() int {
	return len(c.rules)
}
