package ast

import (
	"strconv"

	"strings"

	"fmt"

	"code.justin.tv/amzn/C7-go/internal/parser"
	"code.justin.tv/amzn/C7-go/internal/rule"
	"errors"
	"github.com/antlr/antlr4/runtime/Go/antlr"
	"time"
)

const (
	// OperatorEqual when we have an equal operator
	OperatorEqual = "="
	// OperatorAppend when we have an append operator
	OperatorAppend = "+="
)

// C7Listener is the event handler for the event based parsing done by ANTLR. Events are fired
// as different nodes are reached while parsing. This is very similar to a SAX based xml parser.
type C7Listener struct {
	Rules    rule.Rules // The tree of objects we are parsing into
	Rule     *rule.Rule // The current rule we are filling in
	Selector []string   // The most recent selector
	Errors   []error    // if we encountered any errors
}

// VisitTerminal is called when a terminal node is visited.
func (s *C7Listener) VisitTerminal(node antlr.TerminalNode) {}

// VisitErrorNode is called when an error node is visited.
func (s *C7Listener) VisitErrorNode(node antlr.ErrorNode) {}

// EnterEveryRule is called when any rule is entered.
func (s *C7Listener) EnterEveryRule(ctx antlr.ParserRuleContext) {}

// ExitEveryRule is called when any rule is exited.
func (s *C7Listener) ExitEveryRule(ctx antlr.ParserRuleContext) {}

// EnterFile is called when a file is entered.
func (s *C7Listener) EnterFile(ctx *parser.FileContext) {}

// EnterRecord is called when a record is entered.
func (s *C7Listener) EnterRecord(ctx *parser.RecordContext) {
	// Create an empty rule to populate
	s.Rule = &rule.Rule{
		Keys:      nil,
		Selectors: nil,
		Namespace: "",
		Operator:  "",
	}

	// If we are an empty rule, just default everything
	if ctx.Selectors() == nil {
		return
	}

	s.Selector = []string{}
	s.Rule.Keys = []string{}
}

// ExitRecord is called when a record is exited.
func (s *C7Listener) ExitRecord(ctx *parser.RecordContext) {
	if len(s.Rule.Keys) == 0 {
		return
	}
	s.Rule.Operator = ctx.OPERATOR().GetText()

	s.Rules = append(s.Rules, s.Rule)
	if s.Rule.Selectors == nil {
		s.Rule.Selectors = s.Selector
	}
}

// EnterSelectors is called when selectors are entered.
func (s *C7Listener) EnterSelectors(ctx *parser.SelectorsContext) {}

// ExitSelectors is called when selectors are exited.
func (s *C7Listener) ExitSelectors(ctx *parser.SelectorsContext) {
	if s.Rule.Selectors == nil {
		s.Rule.Selectors = []string{}
	}
	for _, selector := range ctx.AllSelector() {
		s.Rule.Selectors = append(s.Rule.Selectors, selector.GetText())
		s.Selector = append(s.Selector, selector.GetText())
	}
}

// EnterSelector is called when a selector is entered.
func (s *C7Listener) EnterSelector(ctx *parser.SelectorContext) {}

// ExitSelector is called when a selector is exited.
func (s *C7Listener) ExitSelector(ctx *parser.SelectorContext) {}

// EnterKeyspec is called when a keyspec is entered.
func (s *C7Listener) EnterKeyspec(ctx *parser.KeyspecContext) {}

// ExitKeyspec is called when a keyspec is exited.
func (s *C7Listener) ExitKeyspec(ctx *parser.KeyspecContext) {}

// EnterNamespace is called when a namespace is entered.
func (s *C7Listener) EnterNamespace(ctx *parser.NamespaceContext) {
	s.Rule.Namespace = ctx.GetText()
}

// ExitNamespace is called when a namespace is exited.
func (s *C7Listener) ExitNamespace(ctx *parser.NamespaceContext) {}

// EnterKeys is called when keys are entered.
func (s *C7Listener) EnterKeys(ctx *parser.KeysContext) {}

// ExitKeys is called when keys are exited.
func (s *C7Listener) ExitKeys(ctx *parser.KeysContext) {
	s.Rule.Keys = append(s.Rule.Keys, ctx.GetText())
}

// getValue converts a AST value into a go value
func (s *C7Listener) getValue(ctx *parser.ValueContext, rule *rule.Rule) {
	if ctx.NUMBER() != nil {
		strI := ctx.NUMBER().GetText()
		i, err := strconv.ParseInt(strI, 10, 64)
		if err != nil {
			s.Errors = append(s.Errors, err)
		}
		rule.Value.IntValue = &i
	} else if ctx.STRING() != nil {
		txt := ctx.STRING().GetText()
		// trim off the quotes
		txt = txt[1 : len(txt)-1]

		rule.Value.StrValue = &txt
	} else if ctx.BOOL() != nil {
		txt := ctx.BOOL().GetText()
		txt = strings.ToLower(txt)
		tValue := false
		if txt == "true" {
			tValue = true
		}
		rule.Value.BoolValue = &tValue
	} else if ctx.DURATION() != nil {
		txt := ctx.DURATION().GetText()
		d, err := time.ParseDuration(txt)
		if err != nil {
			s.Errors = append(s.Errors, err)
			return
		}
		rule.Value.DurationValue = &d
	}
}

// EnterValue is called when a value is entered.
func (s *C7Listener) EnterValue(ctx *parser.ValueContext) {
	s.getValue(ctx, s.Rule)
}

// ExitValue is called when a value is exited.
func (s *C7Listener) ExitValue(ctx *parser.ValueContext) {}

// ExitFile is called when a file is exited.
func (s *C7Listener) ExitFile(ctx *parser.FileContext) {
}

// recordingErrorListener aggregates errors happening during the lex or parse stages for later retrieval
type recordingErrorListener struct {
	Errors []string
}

func (e *recordingErrorListener) SyntaxError(recognizer antlr.Recognizer, offendingSymbol interface{}, line, column int, msg string, re antlr.RecognitionException) {
	e.Errors = append(e.Errors, fmt.Sprintf("line %d:%d, %s", line, column, msg))
}
func (e *recordingErrorListener) ReportAmbiguity(recognizer antlr.Parser, dfa *antlr.DFA, startIndex, stopIndex int, exact bool, ambigAlts *antlr.BitSet, configs antlr.ATNConfigSet) {
}
func (e *recordingErrorListener) ReportAttemptingFullContext(recognizer antlr.Parser, dfa *antlr.DFA, startIndex, stopIndex int, conflictingAlts *antlr.BitSet, configs antlr.ATNConfigSet) {
}
func (e *recordingErrorListener) ReportContextSensitivity(recognizer antlr.Parser, dfa *antlr.DFA, startIndex, stopIndex, prediction int, configs antlr.ATNConfigSet) {
}

// Parse takes a string and parses it to Rules
func Parse(data string) (rule.Rules, error) {
	// This is pretty much directly out of the ANTLR Golang examples
	lexer := parser.NewC7Lexer(antlr.NewInputStream(data))

	stream := antlr.NewCommonTokenStream(lexer, 0)
	stream.Fill()

	acParser := parser.NewC7Parser(stream)

	errorListener := &recordingErrorListener{}
	acParser.AddErrorListener(errorListener)

	tree := acParser.File()

	if len(errorListener.Errors) > 0 {
		strError := fmt.Sprintf(`The following errors prevented c7 parsing:
  %s`, strings.Join(errorListener.Errors, "\n  "))
		return nil, errors.New(strError)
	}

	walker := antlr.NewParseTreeWalker()

	listener := &C7Listener{
		Rules: []*rule.Rule{},
	}

	// convert the parsed ast to our objects
	walker.Walk(listener, tree)

	if len(listener.Errors) > 0 {
		strError := "Errors encountered parsing: \n"
		for _, err := range listener.Errors {
			strError += fmt.Sprintf("\t%s\n", err.Error())
		}
	}

	return listener.Rules, nil
}
