package loggers

import (
	"code.justin.tv/amzn/TwitchLogging"
)

// Decorative allows users to create a logger that appends keys/values to log statements (decorates log statements
// with additional maintained key, value pairs). This logger calls log on the wrapped logger, and sends the wrapped
// logger all pairs it has maintained. The wrapped logger therefore has the original message, and a combined list
// of the maintained pairs and the keyvals provided in the Log.
type Decorative struct {
	Logger          logging.Logger
	maintainedPairs []*Pair
}

// Pair contains the key, value pairs that the logger maintains, ensuring that each key has an associated value. The
// Decorative logger stores these pairs and sends them all to the wrapped logger.
type Pair struct {
	Key   string
	Value interface{}
}

// Log calls Log() on the wrapped logger, appending the Decorative logger's maintained Pairs to the keyvals provided
func (l *Decorative) Log(msg string, keyvals ...interface{}) {

	// Only log if the logger and wrapped logger exists
	if l == nil || l.Logger == nil {
		return
	}

	// If there aren't any maintainedPairs, just log the msg and keyvals
	if len(l.maintainedPairs) == 0 {
		l.Logger.Log(msg, keyvals...)
		return
	}

	// Otherwise, there are at least some maintained pairs that we need to convert for the wrapped logger
	// Determine the length of the final decorate keyvals list
	totalKeyvals := len(l.maintainedPairs)*2 + len(keyvals)
	// Construct the keyvals slice
	decorativeKeyvals := make([]interface{}, 0, totalKeyvals)
	for _, pair := range l.maintainedPairs {
		decorativeKeyvals = append(decorativeKeyvals, pair.Key, pair.Value)
	}
	// Add the keyvals provided
	decorativeKeyvals = append(decorativeKeyvals, keyvals...)

	// Send the wrapped logger the appropriate message and all keyvals
	l.Logger.Log(msg, decorativeKeyvals...)
}

// With returns a new Decorative logger that additionally maintains the provided key, value pairs
func With(l logging.Logger, key string, val interface{}) *Decorative {
	// Construct a pair from the provided key and value
	pair := &Pair{
		Key:   key,
		Value: val,
	}

	return WithPairs(l, pair)
}

// WithPairs returns a new Decorative logger that additionally maintains the provided pairs
func WithPairs(l logging.Logger, pairs ...*Pair) *Decorative {
	// If we already have a decorative logger, we should append the provided pair to it
	if decorative, ok := l.(*Decorative); ok {
		return &Decorative{
			Logger:          decorative.Logger,
			maintainedPairs: addPairArrays(decorative.maintainedPairs, pairs),
		}
	} else {
		// Otherwise, we need to create a new decorative logger that maintains the provided pair
		return &Decorative{
			Logger:          l,
			maintainedPairs: pairs,
		}
	}
}

// addPairArrays will add two pair arrays.  Note that we CANNOT append(a, b...) because this will use the extra capacity
// of a and can run into race conditions if called by two go routines at the same time
func addPairArrays(a, b []*Pair) []*Pair {
	// If possible, return quickly instead of creating a new slice
	if len(a) == 0 && len(b) == 0 {
		return nil
	} else if len(a) == 0 {
		return b
	} else if len(b) == 0 {
		return a
	}

	n := len(a) + len(b)
	ret := make([]*Pair, 0, n)
	ret = append(ret, a...)
	ret = append(ret, b...)
	return ret
}
