package simplebootstrapecs

import (
	"errors"
	"os"

	"code.justin.tv/amzn/TwitchProcessIdentifier"

	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrap"
	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangBootstrap/bootstrap"
)

// ECS understands how to create the FultonBootstrap object for the ECS platform
// serviceName: the name of the service you are running
// bootstrapConfig: the configuration for bootstrapping
func ECS(serviceName string, customMembers simplebootstrap.FultonCustomMembers, configSetLoader bootstrap.ConfigSetLoader) (*bootstrap.FultonBootstrap, error) {
	// Determine the process identifiers (required for C7)
	tPid, err := GetECSProcessIdentifiers(serviceName)
	if err != nil {
		return nil, err
	}
	if tPid == nil {
		return nil, errors.New("cannot use empty process identifier")
	}

	return simplebootstrap.CommonSimpleBootstrap(tPid, customMembers, configSetLoader)
}

// GetECSProcessIdentifiers obtain all process identifiers for ECS
func GetECSProcessIdentifiers(serviceName string) (*identifier.ProcessIdentifier, error) {
	tPid := identifier.ProcessIdentifier{
		Service:  serviceName,
		Stage:    os.Getenv("FULTON_STAGE"),
		Substage: os.Getenv("FULTON_SUBSTAGE"),
		// Region env var provided by ECS
		Region: os.Getenv("AWS_DEFAULT_REGION"),
		// Cluster that ECS is running
		Machine:  os.Getenv("ECS_CLUSTER"),
		LaunchID: os.Getenv("HOSTNAME"),
		Version:  os.Getenv("FULTON_VERSION"),
	}

	// Default fields until we get all consumers to set them
	if tPid.Version == "" {
		tPid.Version = "UNKNOWN"
	}
	if tPid.Stage == "" {
		tPid.Stage = "development"
	}
	if tPid.Substage == "" {
		tPid.Substage = "primary"
	}

	// Error if AWS vars aren't set
	if tPid.Region == "" {
		return nil, errors.New("unable to start, must provide AWS_DEFAULT_REGION envvar to fulton ecs")
	}
	if tPid.Machine == "" {
		return nil, errors.New("unable to start, must provide ECS_CLUSTER envvar to fulton ecs")
	}
	if tPid.LaunchID == "" {
		return nil, errors.New("unable to start, must provide HOSTNAME envvar to fulton ecs")
	}
	return &tPid, nil
}
