package twirputils

import (
	"time"

	"github.com/golang/protobuf/ptypes"
	google_timestamp "github.com/golang/protobuf/ptypes/timestamp"
	google_wrappers "github.com/golang/protobuf/ptypes/wrappers"
)

// WrapString converts a string to a wrapper pointer
func WrapString(s string) *google_wrappers.StringValue {
	return &google_wrappers.StringValue{
		Value: s,
	}
}

// WrapStringPointer converts a string pointer to a wrapper pointer
// and sets the value on the wrapper if the pointer is not nil
func WrapStringPointer(s *string) *google_wrappers.StringValue {
	if s == nil {
		return nil
	}
	return &google_wrappers.StringValue{
		Value: *s,
	}
}

// UnwrapStringValuePointer turns a StringValue pointer into a string pointer
// If the value pointer is nil this will return nil.
func UnwrapStringValuePointer(v *google_wrappers.StringValue) *string {
	if v == nil {
		return nil
	}
	s := v.GetValue()
	return &s
}

// WrapFloat converts a float to a wrapper pointer
func WrapFloat(f float32) *google_wrappers.FloatValue {
	return &google_wrappers.FloatValue{
		Value: f,
	}
}

// WrapFloatPointer converts a float pointer to a wrapper pointer
// and sets the value on the wrapper if the pointer is not nil
func WrapFloatPointer(f *float32) *google_wrappers.FloatValue {
	if f == nil {
		return nil
	}
	return &google_wrappers.FloatValue{
		Value: *f,
	}
}

// UnwrapFloatValuePointer turns a FloatValue pointer into a float32 pointer
// If the value pointer is nil this will return nil.
func UnwrapFloatValuePointer(v *google_wrappers.FloatValue) *float32 {
	if v == nil {
		return nil
	}
	f := v.GetValue()
	return &f
}

// WrapBool converts a bool to a wrapper pointer
func WrapBool(b bool) *google_wrappers.BoolValue {
	return &google_wrappers.BoolValue{
		Value: b,
	}
}

// WrapBoolPointer converts a bool pointer to a wrapper pointer
// and sets the value on the wrapper if the pointer is not nil
func WrapBoolPointer(b *bool) *google_wrappers.BoolValue {
	if b == nil {
		return nil
	}
	return &google_wrappers.BoolValue{
		Value: *b,
	}
}

// UnwrapBoolValuePointer turns a FloatValue pointer into a float32 pointer
// If the value pointer is nil this will return nil.
func UnwrapBoolValuePointer(v *google_wrappers.BoolValue) *bool {
	if v == nil {
		return nil
	}
	b := v.GetValue()
	return &b
}

// WrapTimePointer attempts to turn a time pointer into a Timestamp pointer
// This conversion can fail, returning an error
// If the time pointer is nil, this will return nil
func WrapTimePointer(t *time.Time) (*google_timestamp.Timestamp, error) {
	if t == nil {
		return nil, nil
	}
	return ptypes.TimestampProto(*t)
}

// UnwrapTimestampPointer basically calls a ptypes function to do convert a timestamp to a time
// but is here to keep all twirp <-> go conversions in the same place
func UnwrapTimestampPointer(ts *google_timestamp.Timestamp) (*time.Time, error) {
	t, err := ptypes.Timestamp(ts)
	return &t, err
}
