package hystrix

import (
	"net/http"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/jixwanwang/apiutils"

	"goji.io/middleware"
)

// CreateCircuit injects a hystrix circuit into the middleware stack. Routes can
// override the circuit name by providing a Name() funtion.  Setting up circuit
// timings is still done explicitly through hystrix calls.
func CreateCircuit(defaultName string) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			name := defaultName
			route := middleware.Handler(r.Context())
			if cast, ok := route.(NamedHandler); ok {
				name = cast.Name()
			}
			if err := hystrix.Do(name, func() error { inner.ServeHTTP(w, r); return nil }, nil); err != nil {
				status := http.StatusServiceUnavailable
				if err == hystrix.ErrMaxConcurrency {
					status = http.StatusTooManyRequests
				}
				apiutils.ServeError(w, apiutils.NewErrorResponse(status, err.Error()))
			}
		})
	}
}
