data "aws_acm_certificate" "orchestrator_ssl_certificate" {
  #
  # The domain needs to be a wildcard domain so users of your service can create their own DNS entries
  # pointing to the VPC Endpoints in their own VPCs that connect up to your LB, so SSL hostname validation
  # works.
  # e.g. they will want to create ${vpc_id}.staging.ext-discovery.us-west-2.internal.justin.tv IN CNAME
  #       ${their_vpc_endpoint_hostname}
  # so their services can resolve something that points to the interface in their own account that is connected
  # to your account.
  #
  domain = "*.${var.environment}.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  statuses    = ["ISSUED"]
  most_recent = true
}

module "nginx" {
  source = "git::ssh://git@git-aws.internal.justin.tv/gds/terraform-fargate_service_template?ref=refs/remotes/origin/master"

  ssl_cert_arn           = data.aws_acm_certificate.orchestrator_ssl_certificate.arn
  build_security_group   = data.aws_security_group.default.id
  ingress_security_group = data.terraform_remote_state.vpc_setup.outputs.us_west_2_twitch_subnets_sg_id
  private_subnets        = data.terraform_remote_state.vpc_setup.outputs.us_west_2_private_subnets
  public_subnets         = data.terraform_remote_state.vpc_setup.outputs.us_west_2_public_subnets
  vpc_id                 = data.terraform_remote_state.vpc_setup.outputs.us_west_2_vpc_id
  service                = var.service
  owner                  = var.owner
  component              = var.component
  environment            = var.environment
  account_name           = var.account_name

  # How many containers to spin up behind the NLB
  ecs_min_container_count = "1"
  ecs_max_container_count = "1"
  enable_auto_scaling     = "true"

  # Use EC2 instead of FARGATE
  launch_type            = "EC2"
  ecs_min_instance_count = "1"
  ecs_max_instance_count = "1"

  healthcheck_port     = "8080"
  healthcheck_protocol = "HTTP"
  container_port       = "8443"
  container_protocol   = "TLS"

  nlb_test_hostname = "test.staging.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  # Slack configuration for the lambda that will send messages to a slack channel to request approval
  # for deploys.
  auto_approve_deploys = "true"

  slack_channel     = "#ext-anchors-oncall"
  slack_username    = "orchestratorNginxDeployer"
  slack_webhook_url = var.slack_webhook_url

  whitelisted_vpc_endpoint_users = [
    "arn:aws:iam::510557735000:root",
    "arn:aws:iam::645130450452:root",
    "arn:aws:iam::641044725657:root",
    "arn:aws:iam::628671981940:root",
    "arn:aws:iam::327140220177:root",
  ]
}

module "alarms" {
  source = "../shared"

  pagerduty_integration_url     = "https://events.pagerduty.com/integration/f123acdb6bc640c28315cab5ebcac5e5/enqueue"
  network_packets_low_threshold = "1000"
}

resource "aws_s3_bucket" "nginx-config" {
  bucket = "${var.account_name}-${var.component}-config"
  acl    = "private"
}

resource "aws_s3_bucket_object" "upload-nginx-config" {
  bucket = "${var.account_name}-${var.component}-config"
  key    = "nginx.conf"
  source = "${path.module}/../../nginx.${var.environment}.conf"
  etag   = "${filemd5("${path.module}/../../nginx.${var.environment}.conf")}"
}

# Upload jenkins artifacts to this bucket
output "orchestrator_nginx_jenkins_artifact_bucket" {
  value = module.nginx.jenkins_artifact_bucket_name
}

# And with this filename
output "orchestrator_nginx_jenkins_artifact_name" {
  value = module.nginx.jenkins_artifact_filename
}

# This is the role arn that we will attach iam policies to so the server can use other resources
output "orchestrator_nginx_iam_role_name" {
  value = module.nginx.ecs_task_iam_role_name
}

output "orchestrator_nginx_endpoint_service_name" {
  value = module.nginx.service_endpoint_service_name
}

resource "aws_iam_role_policy_attachment" "attach-s3-access" {
  role       = "${module.nginx.ecs_task_iam_role_name}"
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"
}

##
## -- This is an example of how to attach more IAM policies to the IAM Role that the docker container
##    will run with.
##
#
# resource "aws_iam_policy" "ecs_task_policy" {
#   name = "orchestrator-ecs-task-policy"
#
#   policy = <<POLICY
# {
#     "Version": "2012-10-17",
#     "Statement": [
#       {
#         "Effect": "Allow",
#         "Action": [
#           "dynamodb:*",
#           "cloudwatch:*",
#           "sns:*",
#           "lambda:*",
#           "logs:*",
#           "states:*"
#         ],
#         "Resource": "*"
#       }
#     ]
#   }
# POLICY
# }
#
#
# resource "aws_iam_role_policy_attachment" "test-attach" {
#   role       = "${module.nginx.ecs_task_iam_role_name}"
#   policy_arn = "${aws_iam_policy.ecs_task_policy.arn}"
# }
#
