data "aws_acm_certificate" "orchestrator_ssl_certificate" {
  #
  # The domain needs to be a wildcard domain so users of your service can create their own DNS entries
  # pointing to the VPC Endpoints in their own VPCs that connect up to your LB, so SSL hostname validation
  # works.
  # e.g. they will want to create ${vpc_id}.staging.ext-discovery.us-west-2.internal.justin.tv IN CNAME
  #       ${their_vpc_endpoint_hostname}
  # so their services can resolve something that points to the interface in their own account that is connected
  # to your account.
  #
  domain = "*.${var.environment}.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  statuses    = ["ISSUED"]
  most_recent = true
}

module "nginx" {
  source = "git::ssh://git@git-aws.internal.justin.tv/gds/terraform-fargate_service_template"

  ssl_cert_arn           = data.aws_acm_certificate.orchestrator_ssl_certificate.arn
  build_security_group   = data.aws_security_group.default.id
  ingress_security_group = data.terraform_remote_state.vpc_setup.outputs.us_west_2_twitch_subnets_sg_id
  private_subnets        = data.terraform_remote_state.vpc_setup.outputs.us_west_2_private_subnets
  public_subnets         = data.terraform_remote_state.vpc_setup.outputs.us_west_2_public_subnets
  vpc_id                 = data.terraform_remote_state.vpc_setup.outputs.us_west_2_vpc_id
  service                = var.service
  owner                  = var.owner
  component              = var.component
  environment            = var.environment
  account_name           = var.account_name

  # How many containers to spin up behind the NLB
  ecs_min_container_count = "22"
  ecs_max_container_count = "30"
  enable_auto_scaling     = "true"
  healthcheck_port        = "8080"
  healthcheck_protocol    = "HTTP"
  container_port          = "8443"
  container_protocol      = "TLS"
  nlb_test_hostname       = "test.production.ext-orchestrator.${data.aws_region.current.name}.internal.justin.tv"

  # Use EC2 instead of FARGATE
  launch_type            = "EC2"
  ecs_min_instance_count = "22"
  ecs_max_instance_count = "44"

  # Slack configuration for the lambda that will send messages to a slack channel to request approval
  # for deploys.

  slack_channel     = "#ext-anchors-oncall"
  slack_username    = "orchestratorNginxDeployer"
  slack_webhook_url = var.slack_webhook_url

  # There are no prod users of the ext/orchestrator vpc service endpoint yet
  whitelisted_vpc_endpoint_users = [
    "arn:aws:iam::523543649671:root",
    "arn:aws:iam::641044725657:root",
    "arn:aws:iam::787149559823:root",
    "arn:aws:iam::736332675735:root",
    "arn:aws:iam::028439334451:root",
  ]
}

module "alarms" {
  source = "../shared"

  pagerduty_integration_url = "https://events.pagerduty.com/integration/e3d77b8efa1346358355a90d12ab7975/enqueue"
  network_packets_low_threshold = "10000"
}

resource "aws_s3_bucket" "nginx-config" {
  bucket = "${var.account_name}-${var.component}-config"
  acl    = "private"
}

resource "aws_s3_bucket_object" "upload-nginx-config" {
  bucket = "${var.account_name}-${var.component}-config"
  key    = "nginx.conf"
  source = "${path.module}/../../nginx.${var.environment}.conf"
  etag   = "${filemd5("${path.module}/../../nginx.${var.environment}.conf")}"
}

# Upload jenkins artifacts to this bucket
output "orchestrator_nginx_jenkins_artifact_bucket" {
  value = module.nginx.jenkins_artifact_bucket_name
}

# And with this filename
output "orchestrator_nginx_jenkins_artifact_name" {
  value = module.nginx.jenkins_artifact_filename
}

# This is the role arn that we will attach iam policies to so the server can use other resources
output "orchestrator_nginx_iam_role_name" {
  value = module.nginx.ecs_task_iam_role_name
}

output "orchestrator_nginx_endpoint_service_name" {
  value = module.nginx.service_endpoint_service_name
}

##
## -- connect the alerts sns topic to pagerduty
##
resource "aws_sns_topic_subscription" "alerts" {
  topic_arn              = module.nginx.alerts_sns_topic_arn
  endpoint_auto_confirms = true
  protocol               = "https"
  endpoint               = "https://events.pagerduty.com/integration/${var.pagerduty_integration_key}/enqueue"
}

resource "aws_iam_role_policy_attachment" "attach-s3-access" {
  role       = "${module.nginx.ecs_task_iam_role_name}"
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"
}
